/*
 *
 * Copyright (c) 2004-2006 Kip Macy
 * All rights reserved.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/socket.h>
#include <sys/queue.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <net/if_dl.h>
#include <net/if_media.h>

#include <net/bpf.h>

#include <net/if_types.h>
#include <net/if.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <vm/vm.h>
#include <vm/pmap.h>

#include <machine/clock.h>      /* for DELAY */
#include <machine/bus.h>
#include <machine/resource.h>
#include <machine/frame.h>


#include <sys/bus.h>
#include <sys/rman.h>

#include <machine/intr_machdep.h>

#include <machine/xen-os.h>
#include <machine/hypervisor.h>
#include <machine/hypervisor-ifs.h>
#include <machine/xen_intr.h>
#include <machine/evtchn.h>
#include <machine/xenbus.h>
#include <machine/gnttab.h>
#include <machine/xen-public/memory.h>

#define GRANT_INVALID_REF	0

#define NET_TX_RING_SIZE __RING_SIZE((netif_tx_sring_t *)0, PAGE_SIZE)
#define NET_RX_RING_SIZE __RING_SIZE((netif_rx_sring_t *)0, PAGE_SIZE)


struct netfront_info;
static void xn_txeof(struct netfront_info *);
static void xn_rxeof(struct netfront_info *);
static void network_alloc_rx_buffers(struct netfront_info *);

static void xn_tick_locked(struct netfront_info *);
static void xn_tick(void *);

static void xn_intr(void *);
static void xn_start_locked(struct ifnet *);
static void xn_start(struct ifnet *);
static int  xn_ioctl(struct ifnet *, u_long, caddr_t);
static void xn_ifinit_locked(struct netfront_info *);
static void xn_ifinit(void *);
static void xn_stop(struct netfront_info *);
#ifdef notyet
static void xn_watchdog(struct ifnet *);
#endif

static void show_device(struct netfront_info *sc);
static void netfront_closing(struct xenbus_device *dev);
static void netif_free(struct netfront_info *info);
static int netfront_remove(struct xenbus_device *dev);

static int talk_to_backend(struct xenbus_device *dev, struct netfront_info *info);
static int create_netdev(int handle, struct xenbus_device *dev, struct ifnet **ifp);
static void netif_disconnect_backend(struct netfront_info *info);
static int setup_device(struct xenbus_device *dev, struct netfront_info *info);
static void end_access(int ref, void *page);

/* Xenolinux helper functions */
static void network_connect(struct ifnet *ifp);

static void xn_free_rx_ring(struct netfront_info *);

static void xn_free_tx_ring(struct netfront_info *);

#define virt_to_mfn(x) (vtomach(x) >> PAGE_SHIFT)

/* XXX: This isn't supported in FreeBSD, so ignore it for now. */
#define TASK_UNINTERRUPTIBLE	0
#define INVALID_P2M_ENTRY (~0UL)

/*
 * Mbuf pointers. We need these to keep track of the virtual addresses
 * of our mbuf chains since we can only convert from virtual to physical,
 * not the other way around.  The size must track the free index arrays.
 */
struct xn_chain_data {
	struct mbuf		*xn_tx_chain[NET_TX_RING_SIZE+1];
        struct mbuf		*xn_rx_chain[NET_RX_RING_SIZE+1];
};


static unsigned long rx_pfn_array[NET_RX_RING_SIZE];
static multicall_entry_t rx_mcl[NET_RX_RING_SIZE+1];
static mmu_update_t rx_mmu[NET_RX_RING_SIZE];

struct net_device_stats
{
	unsigned long	rx_packets;		/* total packets received	*/
	unsigned long	tx_packets;		/* total packets transmitted	*/
	unsigned long	rx_bytes;		/* total bytes received 	*/
	unsigned long	tx_bytes;		/* total bytes transmitted	*/
	unsigned long	rx_errors;		/* bad packets received		*/
	unsigned long	tx_errors;		/* packet transmit problems	*/
	unsigned long	rx_dropped;		/* no space in linux buffers	*/
	unsigned long	tx_dropped;		/* no space available in linux	*/
	unsigned long	multicast;		/* multicast packets received	*/
	unsigned long	collisions;

	/* detailed rx_errors: */
	unsigned long	rx_length_errors;
	unsigned long	rx_over_errors;		/* receiver ring buff overflow	*/
	unsigned long	rx_crc_errors;		/* recved pkt with crc error	*/
	unsigned long	rx_frame_errors;	/* recv'd frame alignment error */
	unsigned long	rx_fifo_errors;		/* recv'r fifo overrun		*/
	unsigned long	rx_missed_errors;	/* receiver missed packet	*/

	/* detailed tx_errors */
	unsigned long	tx_aborted_errors;
	unsigned long	tx_carrier_errors;
	unsigned long	tx_fifo_errors;
	unsigned long	tx_heartbeat_errors;
	unsigned long	tx_window_errors;
	
	/* for cslip etc */
	unsigned long	rx_compressed;
	unsigned long	tx_compressed;
};

struct netfront_info
{
	struct ifnet *xn_ifp;

	struct net_device_stats stats;
	unsigned int tx_full;
    
	netif_tx_front_ring_t tx;
	netif_rx_front_ring_t rx;

	struct mtx   tx_lock;
	struct mtx   rx_lock;

	unsigned int handle;
	unsigned int evtchn, irq;

	/* What is the status of our connection to the remote backend? */
#define BEST_CLOSED       0
#define BEST_DISCONNECTED 1
#define BEST_CONNECTED    2
	unsigned int backend_state;

	/* Is this interface open or closed (down or up)? */
#define UST_CLOSED        0
#define UST_OPEN          1
	unsigned int user_state;

	/* Receive-ring batched refills. */
#define RX_MIN_TARGET 32
#define RX_MAX_TARGET NET_RX_RING_SIZE
	int rx_min_target, rx_max_target, rx_target;

	/*
	 * {tx,rx}_skbs store outstanding skbuffs. The first entry in each
	 * array is an index into a chain of free entries.
	 */

	grant_ref_t gref_tx_head;
	grant_ref_t grant_tx_ref[NET_TX_RING_SIZE + 1]; 
	grant_ref_t gref_rx_head;
	grant_ref_t grant_rx_ref[NET_TX_RING_SIZE + 1]; 

	struct xenbus_device *xbdev;
	int tx_ring_ref;
	int rx_ring_ref;
	uint8_t mac[ETHER_ADDR_LEN];
	struct xn_chain_data	xn_cdata;	/* mbufs */
        unsigned short          xn_rx_free_idxs[NET_RX_RING_SIZE+1];
        unsigned short          xn_tx_free_idxs[NET_RX_RING_SIZE+1];
	struct mbuf		*xn_rx_batch;	/* head of the batch queue */
	struct mbuf		*xn_rx_batchtail;
	int			xn_rx_batchlen;	/* how many queued */
	int			xn_if_flags;
	struct callout	        xn_stat_ch;

};


#define XN_LOCK_INIT(_sc, _name) \
        mtx_init(&(_sc)->tx_lock, #_name"_tx", "network transmit lock", MTX_SPIN); \
        mtx_init(&(_sc)->rx_lock, #_name"_rx", "network receive lock", MTX_SPIN);
#define XN_RX_LOCK(_sc)           mtx_lock_spin(&(_sc)->rx_lock)
#define XN_RX_UNLOCK(_sc)         mtx_unlock_spin(&(_sc)->rx_lock)
#define XN_TX_LOCK(_sc)           mtx_lock_spin(&(_sc)->tx_lock)
#define XN_TX_UNLOCK(_sc)         mtx_unlock_spin(&(_sc)->tx_lock)
#define XN_LOCK(_sc)           mtx_lock_spin(&(_sc)->tx_lock); \
                               mtx_lock_spin(&(_sc)->rx_lock); 
#define XN_UNLOCK(_sc)         mtx_unlock_spin(&(_sc)->rx_lock); \
                               mtx_unlock_spin(&(_sc)->tx_lock)
#define XN_LOCK_ASSERT(_sc)    mtx_assert(&(_sc)->rx_lock, MA_OWNED); \
                               mtx_assert(&(_sc)->tx_lock, MA_OWNED);
#define XN_RX_LOCK_ASSERT(_sc)    mtx_assert(&(_sc)->rx_lock, MA_OWNED); 
#define XN_TX_LOCK_ASSERT(_sc)    mtx_assert(&(_sc)->tx_lock, MA_OWNED); 
#define XN_LOCK_DESTROY(_sc)   mtx_destroy(&(_sc)->rx_lock); \
                               mtx_destroy(&(_sc)->tx_lock);

/* Access macros for acquiring freeing slots in xn_free_{tx,rx}_idxs[]. */
#define ADD_ID_TO_FREELIST(_list, _id)             \
    (_list)[(_id)] = (_list)[0];                   \
    (_list)[0]     = (_id);
#define GET_ID_FROM_FREELIST(_list)                \
 ({ unsigned short _id = (_list)[0]; \
    (_list)[0]  = (_list)[_id];                    \
    (unsigned short)_id; })
#define FREELIST_EMPTY(_list, _maxid) 		   \
    ((_list)[0] == (_maxid+1))
#if DEBUG
static char *be_state_name[] = {
	[BEST_CLOSED]       = "closed",
	[BEST_DISCONNECTED] = "disconnected",
	[BEST_CONNECTED]    = "connected",
};
#endif
#define IPRINTK(fmt, args...) \
    printf("[XEN] " fmt, ##args)
#define WPRINTK(fmt, args...) \
    printf("[XEN] " fmt, ##args)
#define DPRINTK(fmt, args...) \
    printf("[XEN] " fmt, ##args)

static __inline struct mbuf* 
makembuf (struct mbuf *buf)
{
	struct mbuf *m = NULL;
	
        MGETHDR (m, M_DONTWAIT, MT_DATA);
	
        if (! m)
		return 0;
	
	M_MOVE_PKTHDR(m, buf);
	
        MCLGET (m, M_DONTWAIT);
	
        m->m_pkthdr.len = buf->m_pkthdr.len;
        m->m_len = buf->m_len;
	m_copydata(buf, 0, buf->m_pkthdr.len, mtod(m,caddr_t) );
	m->m_ext.ext_args = (vm_paddr_t *)vtophys(mtod(m,caddr_t));
	
       	return m;
}

/**
 * Read the 'mac' node at the given device's node in the store, and parse that
 * as colon-separated octets, placing result the given mac array.  mac must be
 * a preallocated array of length ETH_ALEN (as declared in linux/if_ether.h).
 * Return 0 on success, or errno on error.
 */
static int 
xen_net_read_mac(struct xenbus_device *dev, uint8_t mac[])
{
	char *s;
	int i;
	char *e;
	char *macstr = xenbus_read(NULL, dev->nodename, "mac", NULL);
	if (IS_ERR(macstr)) {
		return PTR_ERR(macstr);
	}
	s = macstr;
	for (i = 0; i < ETHER_ADDR_LEN; i++) {
		mac[i] = strtoul(s, &e, 16);
		if (s == e || (e[0] != ':' && e[0] != 0)) {
			free(macstr, M_DEVBUF);
			return ENOENT;
		}
		s = &e[1];
	}
	free(macstr, M_DEVBUF);
	return 0;
}

/**
 * Entry point to this code when a new device is created.  Allocate the basic
 * structures and the ring buffers for communication with the backend, and
 * inform the backend of the appropriate details for those.  Switch to
 * Connected state.
 */
static int 
netfront_probe(struct xenbus_device *dev, const struct xenbus_device_id *id)
{
	int err;
	struct ifnet *ifp;
	struct netfront_info *info;
	unsigned int handle;
	
	err = xenbus_scanf(NULL, dev->nodename, "handle", "%u", &handle);
	if (err != 1) {
		xenbus_dev_fatal(dev, err, "reading handle");
		return err;
	}

	err = create_netdev(handle, dev, &ifp);
	if (err) {
		xenbus_dev_fatal(dev, err, "creating netdev");
		return err;
	}

	info = ifp->if_softc;
	dev->data = info;

	err = talk_to_backend(dev, info);
	if (err) {
		free(info, M_DEVBUF);
		dev->data = NULL;
		return err;
	}
	
	return 0;
}


/**
 * We are reconnecting to the backend, due to a suspend/resume, or a backend
 * driver restart.  We tear down our netif structure and recreate it, but
 * leave the device-layer structures intact so that this is transparent to the
 * rest of the kernel.
 */
static int 
netfront_resume(struct xenbus_device *dev)
{
	struct netfront_info *info = dev->data;
	
	DPRINTK("%s\n", dev->nodename);
	
	netif_disconnect_backend(info);
	return talk_to_backend(dev, info);
}


/* Common code used when first setting up, and when resuming. */
static int 
talk_to_backend(struct xenbus_device *dev, struct netfront_info *info)
{
	const char *message;
	struct xenbus_transaction *xbt;
	int err;
	
	/* Create shared ring, alloc event channel. */
	err = setup_device(dev, info);
	if (err)
		goto out;
	
 again:
	xbt = xenbus_transaction_start();
	if (IS_ERR(xbt)) {
		xenbus_dev_fatal(dev, err, "starting transaction");
		goto destroy_ring;
	}

	err = xenbus_printf(xbt, dev->nodename, "tx-ring-ref","%u",
			    info->tx_ring_ref);
	if (err) {
		message = "writing tx ring-ref";
		goto abort_transaction;
	}
	err = xenbus_printf(xbt, dev->nodename, "rx-ring-ref","%u",
			    info->rx_ring_ref);
	if (err) {
		message = "writing rx ring-ref";
		goto abort_transaction;
	}
	err = xenbus_printf(xbt, dev->nodename,
			    "event-channel", "%u", info->evtchn);
	if (err) {
		message = "writing event-channel";
		goto abort_transaction;
	}

	err = xenbus_printf(xbt, dev->nodename,
			    "state", "%d", XenbusStateConnected);
	if (err) {
		message = "writing frontend XenbusStateConnected";
		goto abort_transaction;
	}

	err = xenbus_transaction_end(xbt, 0);
	if (err) {
		if (err == EAGAIN)
			goto again;
		xenbus_dev_fatal(dev, err, "completing transaction");
		goto destroy_ring;
	}
	
	return 0;
	
 abort_transaction:
	xenbus_transaction_end(xbt, 1);
	xenbus_dev_fatal(dev, err, "%s", message);
 destroy_ring:
	netif_free(info);
 out:
	return err;
}


static int 
setup_device(struct xenbus_device *dev, struct netfront_info *info)
{
	netif_tx_sring_t *txs;
	netif_rx_sring_t *rxs;
	int err;
	struct ifnet *ifp;
	
	ifp = info->xn_ifp;

	info->tx_ring_ref = GRANT_INVALID_REF;
	info->rx_ring_ref = GRANT_INVALID_REF;
	info->rx.sring = NULL;
	info->tx.sring = NULL;
	info->irq = 0;

	txs = (netif_tx_sring_t *)malloc(PAGE_SIZE, M_DEVBUF, M_NOWAIT);
	if (!txs) {
		err = ENOMEM;
		xenbus_dev_fatal(dev, err, "allocating tx ring page");
		goto fail;
	}
	rxs = (netif_rx_sring_t *)malloc(PAGE_SIZE, M_DEVBUF, M_NOWAIT);
	if (!rxs) {
		err = ENOMEM;
		xenbus_dev_fatal(dev, err, "allocating rx ring page");
		goto fail;
	}
	memset(txs, 0, PAGE_SIZE);
	memset(rxs, 0, PAGE_SIZE);
	info->backend_state = BEST_DISCONNECTED;

	SHARED_RING_INIT(txs);
	FRONT_RING_INIT(&info->tx, txs, PAGE_SIZE);

	SHARED_RING_INIT(rxs);
	FRONT_RING_INIT(&info->rx, rxs, PAGE_SIZE);

	err = xenbus_grant_ring(dev, virt_to_mfn(txs));
	if (err < 0)
		goto fail;
	info->tx_ring_ref = err;

	err = xenbus_grant_ring(dev, virt_to_mfn(rxs));
	if (err < 0)
		goto fail;
	info->rx_ring_ref = err;

	err = xenbus_alloc_evtchn(dev, &info->evtchn);
	if (err)
		goto fail;

	network_connect(ifp);

	info->irq = bind_evtchn_to_irqhandler(
			  info->evtchn, "xn", xn_intr, info, INTR_TYPE_NET | INTR_MPSAFE, NULL);

	show_device(info);
	
	return 0;
	
 fail:
	netif_free(info);
	return err;
}


/**
 * Callback received when the backend's state changes.
 */
static void backend_changed(struct xenbus_device *dev,
			    XenbusState backend_state)
{
	DPRINTK("\n");
	
	switch (backend_state) {
	case XenbusStateInitialising:
	case XenbusStateInitWait:
	case XenbusStateInitialised:
	case XenbusStateConnected:
	case XenbusStateUnknown:
	case XenbusStateClosed:
		break;
		
	case XenbusStateClosing:
		netfront_closing(dev);
		break;
	}
}




static void
xn_free_rx_ring(struct netfront_info *sc)
{
#if 0
	int i;
	
	for (i = 0; i < NET_RX_RING_SIZE; i++) {
		if (sc->xn_cdata.xn_rx_chain[i] != NULL) {
			m_freem(sc->xn_cdata.xn_rx_chain[i]);
			sc->xn_cdata.xn_rx_chain[i] = NULL;
		}
	}
	
	sc->rx.rsp_cons = 0;
	sc->xn_rx_if->req_prod = 0;
	sc->xn_rx_if->event = sc->rx.rsp_cons ;
#endif
}

static void
xn_free_tx_ring(struct netfront_info *sc)
{
#if 0
	int i;
	
	for (i = 0; i < NET_TX_RING_SIZE; i++) {
		if (sc->xn_cdata.xn_tx_chain[i] != NULL) {
			m_freem(sc->xn_cdata.xn_tx_chain[i]);
			sc->xn_cdata.xn_tx_chain[i] = NULL;
		}
	}
	
	return;
#endif
}

static void
network_alloc_rx_buffers(struct netfront_info *sc)
{
	unsigned short id;
	struct mbuf *m_new, *next;
	int i, batch_target;
	RING_IDX req_prod;
	struct xen_memory_reservation reservation;
	grant_ref_t ref;

	req_prod = sc->rx.req_prod_pvt;
	
	if (unlikely(sc->backend_state != BEST_CONNECTED) )
		return;
	
	/*
	 * Allocate skbuffs greedily, even though we batch updates to the
	 * receive ring. This creates a less bursty demand on the memory allocator,
	 * so should reduce the chance of failed allocation requests both for
	 * ourself and for other kernel subsystems.
	 */
	batch_target = sc->rx_target - (req_prod - sc->rx.rsp_cons);
	for (i = sc->xn_rx_batchlen; i < batch_target; i++, sc->xn_rx_batchlen++) {
		MGETHDR(m_new, M_DONTWAIT, MT_DATA);
		if (m_new == NULL) 
			break;
		
		MCLGET(m_new, M_DONTWAIT);
		if (!(m_new->m_flags & M_EXT)) {
			m_freem(m_new);
			break;
		}
		m_new->m_len = m_new->m_pkthdr.len = MCLBYTES;
		
		/* queue the mbufs allocated */
		if (!sc->xn_rx_batch)
			sc->xn_rx_batch = m_new;
		
		if (sc->xn_rx_batchtail)
			sc->xn_rx_batchtail->m_next = m_new;
		sc->xn_rx_batchtail = m_new;
	}
	
	/* Is the batch large enough to be worthwhile? */
	if ( i < (sc->rx_target/2)  )
		return;
	
	for (i = 0, m_new = sc->xn_rx_batch; m_new; 
	     i++, sc->xn_rx_batchlen--, m_new = next) {

		next = m_new->m_next;
		m_new->m_next = NULL;

		m_new->m_ext.ext_args = (vm_paddr_t *)vtophys(m_new->m_ext.ext_buf);
		
		id = GET_ID_FROM_FREELIST(sc->xn_rx_free_idxs);
		KASSERT(id != 0, ("alloc_rx_buffers: found free receive index of 0\n"));
		sc->xn_cdata.xn_rx_chain[id] = m_new;
		
		
		RING_GET_REQUEST(&sc->rx, req_prod + i)->id = id;		
		ref = gnttab_claim_grant_reference(&sc->gref_rx_head);
		PANIC_IF((signed short)ref < 0);
		sc->grant_rx_ref[id] = ref;
		gnttab_grant_foreign_transfer_ref(ref,
						  sc->xbdev->otherend_id);
		RING_GET_REQUEST(&sc->rx, req_prod + i)->gref = ref;
		
		
		rx_pfn_array[i] = vtomach(mtod(m_new,vm_offset_t)) >> PAGE_SHIFT;
		
		/* Remove this page from pseudo phys map before passing back to Xen. */
		xen_phys_machine[((unsigned long)m_new->m_ext.ext_args >> PAGE_SHIFT)] 
			= INVALID_P2M_ENTRY;
	    	
		rx_mcl[i].op = __HYPERVISOR_update_va_mapping;
		rx_mcl[i].args[0] = (unsigned long)mtod(m_new,vm_offset_t);
		rx_mcl[i].args[1] = 0;
		rx_mcl[i].args[2] = 0;
		rx_mcl[i].args[3] = 0;

	} 
	
	KASSERT(i, ("no mbufs processed"));	/* should have returned earlier */
	KASSERT(sc->xn_rx_batchlen == 0, ("not all mbufs processed"));
	sc->xn_rx_batch = sc->xn_rx_batchtail = NULL;
	
	/*
	 * We may have allocated buffers which have entries outstanding
	 in the page * update queue -- make sure we flush those first!  */
	PT_UPDATES_FLUSH();
	
	/* After all PTEs have been zapped we blow away stale TLB entries. */
	rx_mcl[i-1].args[MULTI_UVMFLAGS_INDEX] = UVMF_TLB_FLUSH|UVMF_LOCAL;
	
	/* Give away a batch of pages. */
	rx_mcl[i].op = __HYPERVISOR_memory_op;
	rx_mcl[i].args[0] = XENMEM_decrease_reservation;
	rx_mcl[i].args[1] =  (unsigned long)&reservation;
	
	reservation.extent_start = rx_pfn_array;
	reservation.nr_extents   = i;
	reservation.extent_order = 0;
	reservation.address_bits = 0;
	reservation.domid        = DOMID_SELF;
	
#ifdef notyet
	/* Tell the ballon driver what is going on. */
	balloon_update_driver_allowance(i);
#endif
	
	/* Zap PTEs and give away pages in one big multicall. */
	(void)HYPERVISOR_multicall(rx_mcl, i+1);
	
	/* Check return status of HYPERVISOR_dom_mem_op(). */
	if (unlikely(rx_mcl[i].result != i))
		panic("Unable to reduce memory reservation\n");
	
	/* Above is a suitable barrier to ensure backend will see requests. */
	sc->rx.req_prod_pvt = req_prod + i;
	RING_PUSH_REQUESTS(&sc->rx);
	
	/* Adjust our floating fill target if we risked running out of buffers. */
	if ( ((req_prod - sc->rx.sring->rsp_prod) < (sc->rx_target / 4)) &&
	     ((sc->rx_target *= 2) > sc->rx_max_target) )
		sc->rx_target = sc->rx_max_target;
}

static void
xn_rxeof(struct netfront_info *np)
{
	struct ifnet *ifp;
	netif_rx_response_t  *rx;
	RING_IDX i, rp;
	mmu_update_t *mmu = rx_mmu;
	multicall_entry_t *mcl = rx_mcl;
	struct mbuf *tail_mbuf = NULL, *head_mbuf = NULL, *m, *next;
	unsigned long mfn;
	grant_ref_t ref;


	XN_RX_LOCK_ASSERT(np);
	if (np->backend_state != BEST_CONNECTED)
		return;
	
	ifp = np->xn_ifp;
	
	rp = np->rx.sring->rsp_prod;
	rmb();	/* Ensure we see queued responses up to 'rp'. */
	
	for (i = np->rx.rsp_cons; i != rp; i++) {
		
		rx = RING_GET_RESPONSE(&np->rx, i);
		KASSERT(rx->id != 0, ("xn_rxeof: found free receive index of 0\n"));
		/*
                 * This definitely indicates a bug, either in this driver or
                 * in the backend driver. In future this should flag the bad
                 * situation to the system controller to reboot the backed.
                 */
		if ((ref = np->grant_rx_ref[rx->id]) == GRANT_INVALID_REF) {
			WPRINTK("Bad rx response id %d.\n", rx->id);
			continue;
		}

		
		/* Memory pressure, insufficient buffer headroom, ... */
		if ((mfn = gnttab_end_foreign_transfer_ref(ref)) == 0) {
#if 0
			if (net_ratelimit())
				WPRINTK("Unfulfilled rx req (id=%d, st=%d).\n",
					rx->id, rx->status);
#endif
			RING_GET_REQUEST(&np->rx, np->rx.req_prod_pvt)->id =
				rx->id;
			RING_GET_REQUEST(&np->rx, np->rx.req_prod_pvt)->gref =
				ref;
			np->rx.req_prod_pvt++;
			RING_PUSH_REQUESTS(&np->rx);
			continue;
		}
		
		gnttab_release_grant_reference(&np->gref_rx_head, ref);
		np->grant_rx_ref[rx->id] = GRANT_INVALID_REF;
		
		
		m = (struct mbuf *)np->xn_cdata.xn_rx_chain[rx->id];
		if (m->m_next)
			panic("mbuf is already part of a valid mbuf chain");
		ADD_ID_TO_FREELIST(np->xn_rx_free_idxs, rx->id);
		
		m->m_data += rx->offset;/* (rx->addr & PAGE_MASK); */
		m->m_pkthdr.len = m->m_len = rx->status;
		m->m_pkthdr.rcvif = ifp;
		
		if ( rx->flags & NETRXF_data_validated ) {
			/* Tell the stack the checksums are okay */
			m->m_pkthdr.csum_flags |=
				(CSUM_IP_CHECKED | CSUM_IP_VALID | CSUM_DATA_VALID | CSUM_PSEUDO_HDR);
			m->m_pkthdr.csum_data = 0xffff;
		}

		np->stats.rx_packets++;
		np->stats.rx_bytes += rx->status;
		
		
		/* Remap the page. */
		mmu->ptr = ((vm_offset_t)mfn << PAGE_SHIFT) | MMU_MACHPHYS_UPDATE;
		mmu->val = (unsigned long)m->m_ext.ext_args >> PAGE_SHIFT;
		mmu++;
		/* XXX validate me */
		mcl->op = __HYPERVISOR_update_va_mapping;
		mcl->args[0] = (unsigned long)m->m_data;
		mcl->args[1] = (mfn << PAGE_SHIFT) | PG_V | PG_RW | PG_M | PG_A;
		mcl->args[2] = 0;		
		mcl->args[3] = 0;
		mcl++;

		xen_phys_machine[((unsigned long)m->m_ext.ext_args >> PAGE_SHIFT)] = 
			mfn;

		if (unlikely(!head_mbuf))
			head_mbuf = m;
		
		if (tail_mbuf)
			tail_mbuf->m_next = m;
		tail_mbuf = m;
		
		np->xn_cdata.xn_rx_chain[rx->id] = NULL;
	}
	
	/* Do all the remapping work, and M->P updates,  in one big hypercall. */
	if (likely((mcl - rx_mcl) != 0)) {
		mcl->op = __HYPERVISOR_mmu_update;
		mcl->args[0] = (unsigned long)rx_mmu;
		mcl->args[1] = mmu - rx_mmu;
		mcl->args[2] = 0;
		mcl->args[3] = DOMID_SELF;
		mcl++;
		(void)HYPERVISOR_multicall(rx_mcl, mcl - rx_mcl);
	}
	
	
	/* 
	 * Process all the mbufs after the remapping is complete.
	 * Break the mbuf chain first though.
	 */
	for (m = head_mbuf; m; m = next) {
		next = m->m_next;
		m->m_next = NULL;
		
		ifp->if_ipackets++;
		
		XN_RX_UNLOCK(np);
		
		/* Pass it up. */
		(*ifp->if_input)(ifp, m);
		XN_RX_LOCK(np);
	}
	
	np->rx.rsp_cons = i;
	
	/* If we get a callback with very few responses, reduce fill target. */
	/* NB. Note exponential increase, linear decrease. */
	if (((np->rx.req_prod_pvt - np->rx.sring->rsp_prod) > 
	     ((3*np->rx_target) / 4)) && (--np->rx_target < np->rx_min_target))
		np->rx_target = np->rx_min_target;
	
	network_alloc_rx_buffers(np);

	np->rx.sring->rsp_event = i + 1;

}

static void 
xn_txeof(struct netfront_info *np)
{
	RING_IDX i, prod;
	unsigned short id;
	struct ifnet *ifp;
	struct mbuf *m;
	
	XN_TX_LOCK_ASSERT(np);
	
	if (np->backend_state != BEST_CONNECTED)
		return;
	
	ifp = np->xn_ifp;
	ifp->if_timer = 0;
	
	do {
		prod = np->tx.sring->rsp_prod;
		rmb(); /* Ensure we see responses up to 'rp'. */
		
		for (i = np->tx.rsp_cons; i != prod; i++) {
			id = RING_GET_RESPONSE(&np->tx, i)->id;
			m = np->xn_cdata.xn_tx_chain[id]; 
			
			ifp->if_opackets++;
			KASSERT(m != NULL, ("mbuf not found in xn_tx_chain"));
			M_ASSERTVALID(m);
			if (unlikely(gnttab_query_foreign_access(
					     np->grant_tx_ref[id]) != 0)) {
				printk("network_tx_buf_gc: warning "
				       "-- grant still in use by backend "
				       "domain.\n");
				goto out; 
			}
			gnttab_end_foreign_access_ref(
				np->grant_tx_ref[id], GNTMAP_readonly);
			gnttab_release_grant_reference(
				&np->gref_tx_head, np->grant_tx_ref[id]);
			np->grant_tx_ref[id] = GRANT_INVALID_REF;
			
			np->xn_cdata.xn_tx_chain[id] = NULL;
			ADD_ID_TO_FREELIST(np->xn_tx_free_idxs, id);
			m_freem(m);
		}
		np->tx.rsp_cons = prod;
		
		/*
		 * Set a new event, then check for race with update of tx_cons. Note
		 * that it is essential to schedule a callback, no matter how few
		 * buffers are pending. Even if there is space in the transmit ring,
		 * higher layers may be blocked because too much data is outstanding:
		 * in such cases notification from Xen is likely to be the only kick
		 * that we'll get.
		 */
		np->tx.sring->rsp_event =
			prod + ((np->tx.sring->req_prod - prod) >> 1) + 1;
		
		
		mb();
		
	} while (prod != np->tx.sring->rsp_prod);
	
 out: 
	if (np->tx_full &&
	    ((np->tx.sring->req_prod - prod) < NET_TX_RING_SIZE)) {
		np->tx_full = 0;
#if 0
		if (np->user_state == UST_OPEN)
			netif_wake_queue(dev);
#endif
	}

}

static void
xn_intr(void *xsc)
{
	struct netfront_info *np = xsc;
	struct ifnet *ifp = np->xn_ifp;


	while (np->rx.rsp_cons != np->rx.sring->rsp_prod &&
	       np->user_state == UST_OPEN && 
	       ifp->if_drv_flags & IFF_DRV_RUNNING) {
	
		XN_RX_LOCK(np);
		xn_rxeof(np);	
		XN_RX_UNLOCK(np);
		if (np->tx.rsp_cons != np->tx.sring->rsp_prod) {
			XN_TX_LOCK(np);
			xn_txeof(np);
			XN_TX_UNLOCK(np);			
		}
		if (ifp->if_drv_flags & IFF_DRV_RUNNING &&
		    !IFQ_DRV_IS_EMPTY(&ifp->if_snd))
			xn_start(ifp);
	}
	return;
}

static void
xn_tick_locked(struct netfront_info *sc) 
{
	XN_RX_LOCK_ASSERT(sc);
	callout_reset(&sc->xn_stat_ch, hz, xn_tick, sc);

	/* XXX placeholder for printing debug information */
     
}


static void
xn_tick(void *xsc) 
{
	struct netfront_info *sc;
    
	sc = xsc;
	XN_RX_LOCK(sc);
	xn_tick_locked(sc);
	XN_RX_UNLOCK(sc);
     
}
static void
xn_start_locked(struct ifnet *ifp) 
{
	unsigned short id;
	struct mbuf *m_head, *new_m;
	struct netfront_info *sc;
	netif_tx_request_t *tx;
	RING_IDX i;
	grant_ref_t ref;
	unsigned long mfn, tx_bytes;
	int notify;

	sc = ifp->if_softc;
	tx_bytes = 0;
	

	if (sc->backend_state != BEST_CONNECTED)
		return;
	
	for (i = sc->tx.req_prod_pvt; TRUE; i++) {
		
		IF_DEQUEUE(&ifp->if_snd, m_head);
		if (m_head == NULL) 
			break;
		
		if (FREELIST_EMPTY(sc->xn_tx_free_idxs, NET_TX_RING_SIZE)) {
			IF_PREPEND(&ifp->if_snd, m_head);
			ifp->if_drv_flags |= IFF_DRV_OACTIVE;
			break;
		}
		
		id = GET_ID_FROM_FREELIST(sc->xn_tx_free_idxs);
		
		/*
		 * Start packing the mbufs in this chain into
		 * the fragment pointers. Stop when we run out
		 * of fragments or hit the end of the mbuf chain.
		 */
		new_m = makembuf(m_head);
		tx = RING_GET_REQUEST(&sc->tx, i);
		tx->id = id;
		ref = gnttab_claim_grant_reference(&sc->gref_tx_head);
		PANIC_IF((signed short)ref < 0);
		mfn = virt_to_mfn(mtod(new_m, vm_offset_t));
		gnttab_grant_foreign_access_ref(
			ref, sc->xbdev->otherend_id, mfn, GNTMAP_readonly);
		tx->gref = sc->grant_tx_ref[id] = ref;
		tx->size = new_m->m_pkthdr.len;
#if 0
		tx->flags = (skb->ip_summed == CHECKSUM_HW) ? NETTXF_csum_blank : 0;
#endif
		tx->flags = 0;
		new_m->m_next = NULL;
		new_m->m_nextpkt = NULL;
		
		m_freem(m_head);
		
		sc->xn_cdata.xn_tx_chain[id] = new_m;
		BPF_MTAP(ifp, new_m);

		sc->stats.tx_bytes += new_m->m_pkthdr.len;
		sc->stats.tx_packets++;
	}
	
	sc->tx.req_prod_pvt = i;
	RING_PUSH_REQUESTS_AND_CHECK_NOTIFY(&sc->tx, notify);
	if (notify)
		notify_remote_via_irq(sc->irq);

	xn_txeof(sc);

	if (RING_FULL(&sc->tx)) {
		sc->tx_full = 1;
#if 0
		netif_stop_queue(dev);
#endif
	}



	return;
}    

static void
xn_start(struct ifnet *ifp)
{
	struct netfront_info *sc;
	sc = ifp->if_softc;
	XN_TX_LOCK(sc);
	xn_start_locked(ifp);
	XN_TX_UNLOCK(sc);
}



/* equivalent of network_open() in Linux */
static void 
xn_ifinit_locked(struct netfront_info *sc) 
{
	struct ifnet *ifp;
	
	XN_LOCK_ASSERT(sc);
	
	ifp = sc->xn_ifp;
	
	if (ifp->if_drv_flags & IFF_DRV_RUNNING) 
		return;
	
	xn_stop(sc);
	
	sc->user_state = UST_OPEN;
	
	network_alloc_rx_buffers(sc);
	sc->rx.sring->rsp_event = sc->rx.rsp_cons + 1;
	
	ifp->if_drv_flags |= IFF_DRV_RUNNING;
	ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;
	
	callout_reset(&sc->xn_stat_ch, hz, xn_tick, sc);

}


static void 
xn_ifinit(void *xsc)
{
	struct netfront_info *sc = xsc;
    
	XN_LOCK(sc);
	xn_ifinit_locked(sc);
	XN_UNLOCK(sc);

}


static int
xn_ioctl(struct ifnet *ifp, u_long cmd, caddr_t data)
{
	struct netfront_info *sc = ifp->if_softc;
	struct ifreq *ifr = (struct ifreq *) data;
	int mask, error = 0;
	switch(cmd) {
	case SIOCSIFMTU:
		/* XXX can we alter the MTU on a VN ?*/
#ifdef notyet
		if (ifr->ifr_mtu > XN_JUMBO_MTU)
			error = EINVAL;
		else 
#endif
		{
			ifp->if_mtu = ifr->ifr_mtu;
			ifp->if_drv_flags &= ~IFF_DRV_RUNNING;
			xn_ifinit(sc);
		}
		break;
	case SIOCSIFFLAGS:
		XN_LOCK(sc);
		if (ifp->if_flags & IFF_UP) {
			/*
			 * If only the state of the PROMISC flag changed,
			 * then just use the 'set promisc mode' command
			 * instead of reinitializing the entire NIC. Doing
			 * a full re-init means reloading the firmware and
			 * waiting for it to start up, which may take a
			 * second or two.
			 */
#ifdef notyet
			/* No promiscuous mode with Xen */
			if (ifp->if_drv_flags & IFF_DRV_RUNNING &&
			    ifp->if_flags & IFF_PROMISC &&
			    !(sc->xn_if_flags & IFF_PROMISC)) {
				XN_SETBIT(sc, XN_RX_MODE,
					  XN_RXMODE_RX_PROMISC);
			} else if (ifp->if_drv_flags & IFF_DRV_RUNNING &&
				   !(ifp->if_flags & IFF_PROMISC) &&
				   sc->xn_if_flags & IFF_PROMISC) {
				XN_CLRBIT(sc, XN_RX_MODE,
					  XN_RXMODE_RX_PROMISC);
			} else
#endif
				xn_ifinit_locked(sc);
		} else {
			if (ifp->if_drv_flags & IFF_DRV_RUNNING) {
				xn_stop(sc);
			}
		}
		sc->xn_if_flags = ifp->if_flags;
		XN_UNLOCK(sc);
		error = 0;
		break;
	case SIOCSIFCAP:
		mask = ifr->ifr_reqcap ^ ifp->if_capenable;
		if (mask & IFCAP_HWCSUM) {
			if (IFCAP_HWCSUM & ifp->if_capenable)
				ifp->if_capenable &= ~IFCAP_HWCSUM;
			else
				ifp->if_capenable |= IFCAP_HWCSUM;
		}
		error = 0;
		break;
	case SIOCADDMULTI:
	case SIOCDELMULTI:
#ifdef notyet
		if (ifp->if_drv_flags & IFF_DRV_RUNNING) {
			XN_LOCK(sc);
			xn_setmulti(sc);
			XN_UNLOCK(sc);
			error = 0;
		}
#endif
		/* FALLTHROUGH */
	case SIOCSIFMEDIA:
	case SIOCGIFMEDIA:
		error = EINVAL;
		break;
	default:
		error = ether_ioctl(ifp, cmd, data);
	}
    
	return (error);
}

static void
xn_stop(struct netfront_info *sc)
{	
	struct ifnet *ifp;

	XN_LOCK_ASSERT(sc);
    
	ifp = sc->xn_ifp;

	callout_stop(&sc->xn_stat_ch);

	xn_free_rx_ring(sc);
	xn_free_tx_ring(sc);
    
	ifp->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);
}

/* START of Xenolinux helper functions adapted to FreeBSD */
static void
network_connect(struct ifnet *ifp)
{
	struct netfront_info *np;
	int i, requeue_idx;
	netif_tx_request_t *tx;

	np = ifp->if_softc;
	XN_LOCK(np);

	/* Recovery procedure: */
	
	/* Step 1: Reinitialise variables. */
	np->tx_full = 0;

	/* Step 2: Rebuild the RX and TX ring contents.
	 * NB. We could just free the queued TX packets now but we hope
	 * that sending them out might do some good.  We have to rebuild
	 * the RX ring because some of our pages are currently flipped out
	 * so we can't just free the RX skbs.
	 */

	/* Rebuild the TX buffer freelist and the TX ring itself.
	 * NB. This reorders packets.  We could keep more private state
	 * to avoid this but maybe it doesn't matter so much given the
	 * interface has been down.
	 */
	for (requeue_idx = 0, i = 1; i <= NET_TX_RING_SIZE; i++) {
		if (np->xn_cdata.xn_tx_chain[i] != NULL) {
			struct mbuf *m = np->xn_cdata.xn_tx_chain[i];
			
			tx = RING_GET_REQUEST(&np->tx, requeue_idx);
			requeue_idx++;

			tx->id   = i;
			gnttab_grant_foreign_access_ref(
				np->grant_tx_ref[i], np->xbdev->otherend_id, 
				virt_to_mfn(mtod(m, vm_offset_t)),
				GNTMAP_readonly); 
			tx->gref = np->grant_tx_ref[i];
			tx->size = m->m_pkthdr.len;
			np->stats.tx_bytes += tx->size;
			np->stats.tx_packets++;
		}
	}

	np->tx.req_prod_pvt = requeue_idx;

	/* Rebuild the RX buffer freelist and the RX ring itself. */
	for (requeue_idx = 0, i = 1; i <= NET_RX_RING_SIZE; i++) {
		if (np->xn_cdata.xn_rx_chain[i] == NULL) 
			continue;
		gnttab_grant_foreign_transfer_ref(
			np->grant_rx_ref[i], np->xbdev->otherend_id);
		RING_GET_REQUEST(&np->rx, requeue_idx)->gref =
			np->grant_rx_ref[i];
		RING_GET_REQUEST(&np->rx, requeue_idx)->id = i;
		requeue_idx++; 
	}

	np->rx.req_prod_pvt = requeue_idx;
	RING_PUSH_REQUESTS(&np->rx);


	
	/* Step 3: All public and private state should now be sane.  Get
	 * ready to start sending and receiving packets and give the driver
	 * domain a kick because we've probably just requeued some
	 * packets.
	 */
	np->backend_state = BEST_CONNECTED;
	
	notify_remote_via_irq(np->irq);
	xn_txeof(np);
	
#if 0
	if (np->user_state == UST_OPEN)
		netif_start_queue(dev);
#endif
	XN_UNLOCK(np);		
}


static void 
show_device(struct netfront_info *sc)
{
#if DEBUG
	if (sc) {
		IPRINTK("<vif handle=%u %s(%s) evtchn=%u irq=%u tx=%p rx=%p>\n",
			sc->xn_ifno,
			be_state_name[sc->xn_backend_state],
			sc->xn_user_state ? "open" : "closed",
			sc->xn_evtchn,
			sc->xn_irq,
			sc->xn_tx_if,
			sc->xn_rx_if);
	} else {
		IPRINTK("<vif NULL>\n");
	}
#endif
}

/** Create a network device.
 * @param handle device handle
 */
static int 
create_netdev(int handle, struct xenbus_device *dev, struct ifnet **ifpp)
{
	int i;
	struct netfront_info *np;
	int err;
	struct ifnet *ifp;

	np = (struct netfront_info *)malloc(sizeof(struct netfront_info), M_DEVBUF, M_WAITOK);
	memset(np, 0, sizeof(struct netfront_info));
	
	np->backend_state = BEST_CLOSED;
	np->user_state    = UST_CLOSED;
	np->handle 	      = handle;
	np->xbdev         = dev;
    
	XN_LOCK_INIT(np, xennetif);
	np->rx_target     = RX_MIN_TARGET;
	np->rx_min_target = RX_MIN_TARGET;
	np->rx_max_target = RX_MAX_TARGET;
	
	/* Initialise {tx,rx}_skbs to be a free chain containing every entry. */
	for (i = 0; i <= NET_TX_RING_SIZE; i++) {
		np->xn_tx_free_idxs[i] = (i+1);
		np->grant_tx_ref[i] = GRANT_INVALID_REF;		
	}
	for (i = 0; i <= NET_RX_RING_SIZE; i++) {
		np->xn_rx_free_idxs[i] = (i+1);
		np->grant_rx_ref[i] = GRANT_INVALID_REF;
	}
	/* A grant for every tx ring slot */
	if (gnttab_alloc_grant_references(NET_TX_RING_SIZE,
					  &np->gref_tx_head) < 0) {
		printf("#### netfront can't alloc tx grant refs\n");
		err = ENOMEM;
		goto exit;
	}
	/* A grant for every rx ring slot */
	if (gnttab_alloc_grant_references(NET_RX_RING_SIZE,
					  &np->gref_rx_head) < 0) {
		printf("#### netfront can't alloc rx grant refs\n");
		gnttab_free_grant_references(np->gref_tx_head);
		err = ENOMEM;
		goto exit;
	}
	
	err = xen_net_read_mac(dev, np->mac);
	if (err) {
		xenbus_dev_fatal(dev, err, "parsing %s/mac", dev->nodename);
		goto out;
	}
	
	/* Set up ifnet structure */
	*ifpp = ifp = np->xn_ifp = if_alloc(IFT_ETHER);
    	ifp->if_softc = np;
    	if_initname(ifp, "xn", np->handle /* ifno */);
    	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX;
    	ifp->if_ioctl = xn_ioctl;
    	ifp->if_output = ether_output;
    	ifp->if_start = xn_start;
#ifdef notyet
    	ifp->if_watchdog = xn_watchdog;
#endif
    	ifp->if_init = xn_ifinit;
    	ifp->if_mtu = ETHERMTU;
    	ifp->if_snd.ifq_maxlen = NET_TX_RING_SIZE - 1;
	
#ifdef notyet
    	ifp->if_hwassist = XN_CSUM_FEATURES;
    	ifp->if_capabilities = IFCAP_HWCSUM;
    	ifp->if_capenable = ifp->if_capabilities;
#endif    
	
    	ether_ifattach(ifp, np->mac);
    	callout_init(&np->xn_stat_ch, CALLOUT_MPSAFE);

	return 0;

 exit:
 out:
	panic("do something smart");

}

/**
 * Handle the change of state of the backend to Closing.  We must delete our
 * device-layer structures now, to ensure that writes are flushed through to
 * the backend.  Once is this done, we can switch to Closed in
 * acknowledgement.
 */
static void netfront_closing(struct xenbus_device *dev)
{
#if 0
	struct netfront_info *info = dev->data;

	DPRINTK("netfront_closing: %s removed\n", dev->nodename);

	close_netdev(info);
#endif
	xenbus_switch_state(dev, NULL, XenbusStateClosed);
}


static int netfront_remove(struct xenbus_device *dev)
{
	struct netfront_info *info = dev->data;

	DPRINTK("%s\n", dev->nodename);

	netif_free(info);
	free(info, M_DEVBUF);

	return 0;
}


static void netif_free(struct netfront_info *info)
{
	netif_disconnect_backend(info);
#if 0
	close_netdev(info);
#endif
}



static void netif_disconnect_backend(struct netfront_info *info)
{
	xn_stop(info);
	end_access(info->tx_ring_ref, info->tx.sring);
	end_access(info->rx_ring_ref, info->rx.sring);
	info->tx_ring_ref = GRANT_INVALID_REF;
	info->rx_ring_ref = GRANT_INVALID_REF;
	info->tx.sring = NULL;
	info->rx.sring = NULL;

#if 0
	if (info->irq)
		unbind_from_irqhandler(info->irq, info->netdev);
#else 
	panic("FIX ME");
#endif
	info->evtchn = info->irq = 0;
}


static void end_access(int ref, void *page)
{
	if (ref != GRANT_INVALID_REF)
		gnttab_end_foreign_access(ref, 0, page);
}


/* ** Driver registration ** */


static struct xenbus_device_id netfront_ids[] = {
	{ "vif" },
	{ "" }
};


static struct xenbus_driver netfront = {
	.name = "vif",
	.ids = netfront_ids,
	.probe = netfront_probe,
	.remove = netfront_remove,
	.resume = netfront_resume,
	.otherend_changed = backend_changed,
};

static void
netif_init(void *unused)
{
	if (xen_start_info->flags & SIF_INITDOMAIN)
		return;

	IPRINTK("Initialising virtual ethernet driver.\n");

	xenbus_register_frontend(&netfront);

    
}

SYSINIT(xennetif, SI_SUB_PSEUDO, SI_ORDER_ANY, netif_init, NULL)

/*
 * Local variables:
 * mode: C
 * c-set-style: "BSD"
 * c-basic-offset: 8
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 */
