/*-
 * Copyright (c) 2003 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/fcntl.h>
#include <sys/mman.h>

#include <machine/ioctl_meteor.h>

#include <stdio.h>
#include <unistd.h>

#include "video.h"

static int dev_fd = -1;

int
vid_open(char *devname)
{

	dev_fd = open(devname, O_RDONLY);
	if (dev_fd == -1) {
		perror(devname);
		return (-1);
	}

	return (0);
}

void
vid_close(void)
{

	close(dev_fd);
	dev_fd = -1;
}

ssize_t
vid_read(char *fb, int fblen)
{
	ssize_t len;
	int i;

	i = METEOR_CAP_SINGLE;
	if (ioctl(dev_fd, METEORCAPTUR, &i) < 0) {
		perror("METEORCAPTUR");
		return (-1);
	}

	len = read(dev_fd, fb, fblen);
	if (len == -1)
		perror("read");
	return (len);
}

void *
vid_map(int len)
{
	void *p;
	int i;

	p = mmap(NULL, len, PROT_READ, MAP_SHARED, dev_fd, NULL);
	if (p == MAP_FAILED) {
		perror("mmap");
		return (NULL);
	}

	i = METEOR_CAP_CONTINOUS;
	if (ioctl(dev_fd, METEORCAPTUR, &i) < 0) {
		perror("METEORCAPTUR");
		return (NULL);
	}

	return (p);
}

void
vid_unmap(void *p, int len)
{
	int i;

	munmap(p, len);

	i = METEOR_CAP_STOP_CONT;
	if (ioctl(dev_fd, METEORCAPTUR, &i) < 0)
		perror("METEORCAPTUR");
}

int
vid_setcapturesize(int columns, int rows)
{
	struct meteor_geomet geo;

	geo.rows = rows;
	geo.columns = columns;
	geo.frames = 1;
	geo.oformat = METEOR_GEO_RGB24;

	if (ioctl(dev_fd, METEORSETGEO, &geo) < 0) {
		perror("METEORSETGEO");
		return (-1);
	}
	return (0);
}

int
vid_setinputformat(int format)
{

	if (ioctl(dev_fd, METEORSFMT, &format) < 0) {
		perror("METEORSFMT");
		return (-1);
	}
	return (0);
}

int
vid_setinputsource(int source)
{
	int i;
	
	i = source;
	if (ioctl(dev_fd, METEORSINPUT, &i) < 0) {
		perror("METEORSINPUT");
		return (-1);
	}
	return (0);
}

int
vid_setbrightness(char brightness)
{

	if (ioctl(dev_fd, METEORSBRIG, &brightness) < 0) {
		perror("METEORSBRIG");
		return (-1);
	}
	return (0);
}

int
vid_setcontrast(char contrast)
{

	if (ioctl(dev_fd, METEORSCONT, &contrast) < 0) {
		perror("METEORSCONT");
		return (-1);
	}
	return (0);
}
