/*-
 * Copyright (c) 2003-2004 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/callout.h>
#include <sys/conf.h>
#include <sys/cons.h>
#include <sys/kernel.h>
#include <sys/lock.h>
#include <sys/mbuf.h>
#include <sys/mutex.h>
#include <sys/sysctl.h>
#include <sys/systm.h>
#include <sys/tty.h>
#include <sys/uio.h>

#include <dev/ethercons/ethercons.h>

SYSCTL_DECL(_kern_ethercons);

static d_open_t ethercons_tty_open;
static d_close_t ethercons_tty_close;
static d_ioctl_t ethercons_tty_ioctl;

static cn_probe_t ethercons_cnprobe;
static cn_init_t ethercons_cninit;
static cn_term_t ethercons_cnterm;
static cn_putc_t ethercons_cnputc;

CONS_DRIVER(ethercons, ethercons_cnprobe, ethercons_cninit, ethercons_cnterm,
    NULL, NULL, ethercons_cnputc, NULL);

static struct cdevsw ethercons_cdevsw = {
	.d_version = D_VERSION,
	.d_open = ethercons_tty_open,
	.d_close = ethercons_tty_close,
	.d_read = ttyread,
	.d_write = ttywrite,
	.d_ioctl = ethercons_tty_ioctl,
	.d_poll = ttypoll,
	.d_name = "ethercons",
	.d_flags = D_TTY | D_NEEDGIANT,
	.d_kqfilter = ttykqfilter,
};

static struct tty	*ethercons_tty;
static struct cdev	*ethercons_dev;

/*
 * Because the low level console code cannot call directly into the network
 * stack under cnputc(), we have a small ring buffer between cnputc() and the
 * network transmit routine running from a callout.  ethercons_cnputc() puts
 * characters in the ring and wakes up an ethercons worker thread to process
 * the characters asynchronously.  In the more general tty write case, the
 * tty code can dispatch directly into the stack, which has some ordering and
 * latency implications.
 *
 * XXXRW: There are a gratuitous quantity of variables exported via sysctl.
 * They should probably be trimmed.
 */
#define	ETHERCONS_RING_SIZE	256
static struct mtx	 ethercons_ring_mtx;
static u_int		 ethercons_ring_start, ethercons_ring_finish;
static u_int		 ethercons_ring_len;
static char		 ethercons_ring[ETHERCONS_RING_SIZE];

static u_int		 ethercons_ring_written;
static u_int		 ethercons_ring_read;
static u_int		 ethercons_ring_overflows;

static struct callout	 ethercons_callout;
static int		 ethercons_callouts_per_second = 10;
static int		 ethercons_timer_count;

SYSCTL_UINT(_kern_ethercons, OID_AUTO, ring_len, CTLFLAG_RD,
    &ethercons_ring_len, 0, "");
SYSCTL_UINT(_kern_ethercons, OID_AUTO, ring_written, CTLFLAG_RD,
    &ethercons_ring_written, 0, "");
SYSCTL_UINT(_kern_ethercons, OID_AUTO, ring_read, CTLFLAG_RD,
    &ethercons_ring_read, 0, "");
SYSCTL_UINT(_kern_ethercons, OID_AUTO, ring_overflows, CTLFLAG_RD,
    &ethercons_ring_overflows, 0, "");
SYSCTL_UINT(_kern_ethercons, OID_AUTO, timer_count, CTLFLAG_RD,
    &ethercons_timer_count, 0, "");

static void
ethercons_cnprobe(struct consdev *cp)
{

	cp->cn_pri = CN_REMOTE;
	/* cp->cn_pri = CN_NORMAL; */
	/* cp->cn_pri = CN_DEAD; */
	cp->cn_arg = NULL;
	cp->cn_flags = CN_FLAG_NODEBUG;
	strcpy(cp->cn_name, "ethercons");
}

static void
ethercons_cninit(struct consdev *cp)
{

	mtx_init(&ethercons_ring_mtx, "ethercons_ring", NULL, MTX_SPIN);
}

static void
ethercons_cnterm(struct consdev *cp)
{

}

static void
ethercons_cnputc(struct consdev *cp, int c)
{

	mtx_lock_spin(&ethercons_ring_mtx);
	ethercons_ring[ethercons_ring_finish] = c;
	ethercons_ring_finish = (ethercons_ring_finish + 1) %
	    ETHERCONS_RING_SIZE;
	ethercons_ring_len++;
	ethercons_ring_written++;
	if (ethercons_ring_len > ETHERCONS_RING_SIZE) {
		ethercons_ring_start = (ethercons_ring_start + 1) %
		    ETHERCONS_RING_SIZE;
		ethercons_ring_len--;
		ethercons_ring_overflows++;
	}
	mtx_unlock_spin(&ethercons_ring_mtx);
}

static void
ethercons_ring_drain(void)
{
	char data[2];
	char c;

	/*
	 * XXXRW: Lots and lots of little packets.  Wouldn't a few bigger
	 * ones be nicer?
	 */
	mtx_lock_spin(&ethercons_ring_mtx);
	while (ethercons_ring_len != 0) {
		c = ethercons_ring[ethercons_ring_start];
		ethercons_ring_start = (ethercons_ring_start + 1) %
		    ETHERCONS_RING_SIZE;
		ethercons_ring_len--;
		ethercons_ring_read++;
		mtx_unlock_spin(&ethercons_ring_mtx);

		data[0] = c;
		data[1] = '\0';
		ethercons_transmit(2, data);

		mtx_lock_spin(&ethercons_ring_mtx);
	}
	mtx_unlock_spin(&ethercons_ring_mtx);
}

static void
ethercons_timer(void *arg)
{

	ethercons_timer_count++;

	ethercons_ring_drain();

	callout_reset(&ethercons_callout, hz / ethercons_callouts_per_second,
	    ethercons_timer, NULL);
}

static void
start_ethercons_timer(void *dummy)
{

	/*
	 * With debug.mpsafenet, this can probably be CALLOUT_MPSAFE.
	 */
	callout_init(&ethercons_callout, 0);
	callout_reset(&ethercons_callout, hz / ethercons_callouts_per_second,
	    ethercons_timer, NULL);
}

SYSINIT(start_ethercons_timer, SI_SUB_DRIVERS, SI_ORDER_MIDDLE,
    start_ethercons_timer, NULL);

/*
 * Launch transmission of next packet.  To avoid generting packets that
 * are too large, and so that we can use the stack for a local buffer,
 * we transmit at most MAX_CHUNK in a given packet.  MAX_CHUNK includes
 * the terminating nul.
 */
#define	MAX_CHUNK	128
static void
ethercons_tty_oproc(struct tty *tp)
{
	char data[MAX_CHUNK];
	int i;

	KASSERT(tp == ethercons_tty, ("tp != ethercons_tty"));

	if (tp->t_state & TS_TTSTOP)
		return;
	if (tp->t_state & TS_BUSY)
		return;

	tp->t_state |= TS_BUSY;
	while ((i = q_to_b(&tp->t_outq, data, MAX_CHUNK-1)) > 0) {
		data[i] = '\0';
		ethercons_transmit(i + 1, data);
	}
	tp->t_state &= ~TS_BUSY;
	ttwwakeup(tp);
}
#undef	MAX_CHUNK

static void
ethercons_tty_stop(struct tty *tp, int flag)
{

}

static int
ethercons_tty_param(struct tty *tp, struct termios *t)
{

	KASSERT(tp == ethercons_tty, ("tp != ethercons_tty"));

#if 0
	/* XXX? */
	ttsetwater(tp);
#endif
	return (0);
}

void
ethercons_tty_intr(char c)
{
	struct tty *tp = ethercons_tty;

	/*
	 * Perhaps this should be in a taskqueue of some sort rather than in
	 * the netisr context?
	 */
	mtx_lock(&Giant);
	ttyld_rint(tp, c);
	mtx_unlock(&Giant);
}

static void
ethercons_tty_init(void *arg)
{
	struct tty *tp;

	tp = ethercons_tty = ttymalloc(NULL);

	tp->t_dev = ethercons_dev = make_dev(&ethercons_cdevsw, 0, UID_ROOT,
	    GID_WHEEL, 0600, "ethercons");
	ethercons_dev->si_tty = tp;
	tp->t_oproc = ethercons_tty_oproc;
	tp->t_param = ethercons_tty_param;
	tp->t_stop = ethercons_tty_stop;
}
SYSINIT(ethercons_tty_init, SI_SUB_DRIVERS, SI_ORDER_MIDDLE,
    ethercons_tty_init, NULL);

#if 0
static void
ethercons_tty_detach(void)
{

	/* Order? */
	destroy_dev(ethercons_dev);
	/* ttyfree(tp); */
}
#endif

static int
ethercons_tty_open(struct cdev *dev, int flags, int mode, struct thread *td)
{
	struct tty *tp;
	int error;

	tp = dev->si_tty;
	KASSERT(tp == ethercons_tty, ("tp != ethercons_tty"));

	if (tp->t_state & TS_ISOPEN) {
		if (tp->t_state & TS_XCLUDE) {
			error = suser(td);
			if (error)
				return (EBUSY);
		}
	} else {
		tp->t_cflag = TTYDEF_CFLAG;
		tp->t_iflag = TTYDEF_IFLAG;
		tp->t_lflag = TTYDEF_LFLAG;
		tp->t_oflag = TTYDEF_OFLAG;
		tp->t_ispeed = tp->t_ospeed = TTYDEF_SPEED;
		ttychars(tp);
		ttyld_modem(tp, 1);
	}
	error = ttyopen(dev, tp);
	if (error)
		return (error);
	return (ttyld_open(tp, dev));
}

static int
ethercons_tty_close(struct cdev *dev, int flags, int mode, struct thread *td)
{
	struct tty *tp;

	tp = dev->si_tty;
	KASSERT(tp == ethercons_tty, ("tp != ethercons_tty"));

	ttyld_close(tp, flags);
	ttyclose(tp);
	return (0);
}

static int
ethercons_tty_ioctl(struct cdev *dev, u_long cmd, caddr_t data, int flags,
    struct thread *td)
{
	struct tty *tp;

	tp = dev->si_tty;
	KASSERT(tp == ethercons_tty, ("tp != ethercons_tty"));

	return (ttyioctl(dev, cmd, data, flags, td));
}
