/*-
 * Copyright (c) 2003 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <net/bpf.h>
#include <net/ethernet.h>
#include <net/if.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bpf.h"

static struct bpf_insn	instructions[] = {
	/* (000) ldh	[12]				*/
	{ 0x28, 0, 0, 0x0000000c },
	/* (001) jeq	#0x666	   jt 2    jf 3	*/
	{ 0x15, 0, 1, 0x00000666 },
	/* (002) ret	#96				*/
	{ 0x6, 0, 0, 0x00000060 },
	/* (003) ret	#0				*/
	{ 0x6, 0, 0, 0x00000000 },
};
static int	num_instructions = sizeof(instructions) /
		    sizeof(struct bpf_insn);

static u_char	*bpf_buffer;
static u_char	*bp, *ep;
static int	 bpf_buffer_len;

static int	 bpf_fd = -1;

int
bpf_open(int bufsize, char *ifname)
{
	char device[sizeof("/dev/bpf000")];
	struct bpf_program bpf;
#if 0
	struct timeval tv;
#endif
	struct ifreq ifr;
	u_int data;
	int fd, n;

	if (bufsize == 0)
		bufsize = 32768;

	if (strlen(ifname)+1 > sizeof(ifr.ifr_name)) {
		errno = EINVAL;
		return (-1);
	}

	if (bpf_buffer != NULL || bpf_fd != -1) {
		errno = EBUSY;
		return (-1);
	}

	bpf_buffer = malloc(bufsize);
	if (bpf_buffer == NULL)
		return (-1);
	bpf_buffer_len = bufsize;
	bp = ep = bpf_buffer;

	n = 0;
	do {
		sprintf(device, "/dev/bpf%d", n++);
		fd = open(device, O_RDWR);
	} while (fd < 0 && errno == EBUSY && n < 1000);

	if (fd < 0) {
		free(bpf_buffer);
		bpf_buffer = NULL;
		return (-1);
	}

	if (ioctl(fd, BIOCSBLEN, &bufsize)) {
		free(bpf_buffer);
		bpf_buffer = NULL;
		close(fd);
		return (-1);
	}

	data = 1;
	if (ioctl(fd, BIOCIMMEDIATE, &data) < 0) {
		perror("BIOCSRTIMEOUT");
		close(fd);
		return (-1);
	}
#if 0
	tv.tv_sec = 1;
	tv.tv_usec = 0;
	if (ioctl(fd, BIOCSRTIMEOUT, &tv) < 0) {
		perror("BIOCSRTIMEOUT");
		close(fd);
		return (-1);
	}
#endif

	strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
	if (ioctl(fd, BIOCSETIF, &ifr)) {
		free(bpf_buffer);
		bpf_buffer = NULL;
		close(fd);
		return (-1);
	}

	bpf.bf_len = num_instructions;
	bpf.bf_insns = instructions;
	if (ioctl(fd, BIOCSETF, &bpf)) {
		free(bpf_buffer);
		bpf_buffer = NULL;
		close(fd);
		return (-1);
	}

	bpf_fd = fd;
	return (0);
}

void
bpf_close(void)
{

	if (bpf_fd != -1) {
		close(bpf_fd);
		bpf_fd = -1;
	}

	if (bpf_buffer != NULL) {
		free(bpf_buffer);
		bpf_buffer = NULL;
	}
}

void
bpf_freepacket(struct packet *pkt)
{

	free(pkt);
}

struct packet *
bpf_nextpacket(void)
{
	struct bpf_hdr *hdr;
	struct packet *pkt;
	size_t len;

	pkt = malloc(sizeof(*pkt));
	if (pkt == NULL)
		return (NULL);

	do {
		if (bp >= ep) {
			do {
				len = read(bpf_fd, bpf_buffer, bpf_buffer_len);
			} while (len == 0 || (len == -1 && errno == EINTR));
			if (len == -1)
				return (NULL);
			bp = bpf_buffer;
			ep = bp + len;
		}

		hdr = (struct bpf_hdr *) bp;
		bp += BPF_WORDALIGN(hdr->bh_caplen + hdr->bh_hdrlen);
	} while (hdr->bh_caplen > PACKET_MAXLEN);

	pkt->p_len = hdr->bh_caplen;
	bcopy((char *)hdr + hdr->bh_hdrlen, pkt->p_data, pkt->p_len);
	return (pkt);
}

int
bpf_sendpacket(char *packet, int len)
{

	return (write(bpf_fd, packet, len));
}
