/*-
 * Copyright (c) 1999 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *       $Id: $
 */

/*
 * User-land audit daemon
 * Listens on /dev/audit reading out records, and dropping them into 
 * /var/log/audit.
 * Disables its own auditing to prevent nasties.
 */

#include <sys/types.h>
#include <sys/audit.h>
#include <sys/fcntl.h>
#include <sys/uio.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "auditd.h"

#define LOGFILE	"/var/log/audit.log"

void
usage(void)
{

	fprintf(stderr, "auditd [-d] [-f auditdevice] [-l logfile]\n");
	exit(-1);
}

aud_rec_t
get_a_valid_record(int filedes)
{
	aud_info_t	info;
	aud_hdr_t	hdr;
	aud_rec_t	ar;
	uid_t	uid;
	int	i;

	ar = aud_read_sc(filedes, &uid);
	if (!ar) {
		perror("aud_read_sc");
		return(NULL);
	}

	i = aud_valid(ar);
	if (i == -1) {
		perror("aud_valid");
		aud_free(ar);
		return(NULL);
	}

	i = aud_get_hdr(ar, 1, &hdr);
	if (i == -1) {
		perror("aud_get_hdr");
		aud_free(ar);
		return(NULL);
	}

	i = aud_get_hdr_info(hdr, AUD_AUD_ID, &info);
	if (i == -1) {
		/* perror("aud_get_hdr_info"); */
		info.aud_info_type = AUD_TYPE_AUD_ID;
		info.aud_info_length = sizeof(uid_t);
		info.aud_info_p = &uid;
		i = aud_put_hdr_info(hdr, AUD_EVENT_TYPE, AUD_AUD_ID, &info);
		if (i == -1) {
			perror("aud_put_hdr_info");
			aud_free(ar);
			return(NULL);
		}
	} else {
		if ((info.aud_info_type != AUD_TYPE_AUD_ID) ||
		    (info.aud_info_length != sizeof(aud_id_t)) ||
		    (*((uid_t *) info.aud_info_p) != uid)) {
			fprintf(stderr, "info: bad uid_t\n");
			aud_free(ar);
			return(NULL);
		}
	}

	return(ar);
}


/*
 * Some sort of policy file should be read to determine the accept rule,
 * possibly adapting to events themselves.
 */
int
aud_policy_accept(aud_rec_t ar)
{
	aud_hdr_t	hdr;
	aud_info_t	info;
	int	i;

	i = aud_get_hdr(ar, 1, &hdr);
	if (i == -1)
		return(0);

	i = aud_get_hdr_info(hdr, AUD_EVENT_TYPE, &info);
	if (i == -1)
		return(0);

	switch(info.aud_info_type) {
	case AUD_TYPE_INT:	/* kernel audit record for syscall event */
		switch(*(int *)info.aud_info_p) {
			case AUD_AET_AUD_SWITCH:
			case AUD_AET_AUD_WRITE:
			case AUD_AET_CHMOD:
			case AUD_AET_CHOWN:
			case AUD_AET_CREAT:
			case AUD_AET_EXEC:
			case AUD_AET_EXIT:
			case AUD_AET_FORK:
			case AUD_AET_LINK:
			case AUD_AET_MKDIR:
			case AUD_AET_MKFIFO:
			case AUD_AET_OPEN:
			case AUD_AET_RENAME:
			case AUD_AET_SETGID:
			case AUD_AET_SETUID:	
			case AUD_AET_UNLINK:
			case AUD_AET_UTIME:
				return(1);
			default:
				return(0);
		}
	case AUD_TYPE_STRING:	/* user audit record for user event */
		return(1);
	default:
		return(0);
	}
}


void
main(int argc, char *argv[])
{
	aud_rec_t	ar;
	int	i, devaudit_fd, auditlog_fd, size, bufsize, not_daemon;
	char	ch, *buf, *devfile=AUDIT_DEVICE, *logfile=LOGFILE;
	uid_t	uid;

	not_daemon = 0;
	while ((ch = getopt(argc, argv, "df:l:")) != -1) {
		switch(ch) {
		case 'd':
			not_daemon = 1;
			break;
		case 'f':
			devfile = optarg;
			break;
		case 'l':
			logfile = optarg;
			break;
		default:
			/* warnx("unknown flag -%c ignored", optopt); */
			usage();
		}
	}

	buf = (char *) malloc(AUDIT_DEF_BUF_SIZE * sizeof(char));
	if (!buf) {
		fprintf(stderr, "auditd: unable to malloc buffer\n");
		exit(-1);
	}
	bufsize = AUDIT_DEF_BUF_SIZE;

	i = aud_switch(AUD_STATE_OFF);
	if (i == -1) {
		perror("auditd: aud_switch");
		exit(-1);
	}

	devaudit_fd = open(devfile, O_RDONLY);
	if (devaudit_fd == -1) {
		perror("auditd: open: audit device");
		exit(-1);
	}

	auditlog_fd = open(LOGFILE, O_WRONLY | O_APPEND | O_CREAT, 0600);
	if (auditlog_fd == -1) {
		perror("auditd: open: audit log");
		exit(-1);
	}

	if (!not_daemon) {
		i = daemon(1, 1);
		if (i == -1) {
			perror("auditd: daemon");
			exit(-1);
		}
	}

	while (1) {
		ar = get_a_valid_record(devaudit_fd);
		if (ar) {
			if (aud_policy_accept(ar)) {
				i = aud_write(auditlog_fd, ar);
				if (i == -1) {
					perror("aud_write");
				}
				aud_free(ar);
			}
		}
	}
}
