/*
 * Copyright (c) 1999 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *       $Id: $
 */

/*
 * Support for POSIX.1e.AUD auditing support
 */

#include "opt_devfs.h"
#include "opt_posix.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/proc.h>
#include <sys/vnode.h>
#include <sys/filio.h>
#include <sys/ttycom.h>
#include <sys/signalvar.h>
#include <sys/syslog.h>
#include <sys/sysproto.h>
#include <sys/kernel.h>
#include <sys/poll.h>
#include <sys/filedesc.h>
#include <sys/malloc.h>
#ifdef DEVFS
#include <sys/devfsext.h>
#endif /*DEVFS*/
#include <sys/audit.h>

#ifdef _POSIX_AUD
#define AUDIT_RDPRI	(PZERO + 1)

#define AUDIT_ASYNC	0x04
#define AUDIT_RDWAIT	0x08

#define CDEV_MAJOR 120

#define	MIN(a,b)	(((a)<(b))?(a):(b))

#define AUDIT_MAX_AUD_WRITE_LEN	4096

/*
 * Prototypes
 */
static void
audit_init(void *unused);
/*
static int
audit_device_close(dev_t dev, int flags, int fmt, struct proc *p);
static int
audit_device_open(dev_t dev, int flags, int fmt, struct proc *p);
static int
audit_device_read(dev_t dev, struct uio *uio, int flags);
static int
audit_device_ioctl(dev_t dev, u_long cmd, caddr_t data, int flags,
	struct proc *p);
static int
audit_device_poll(dev_t dev, int events, struct proc *p);
*/

static d_open_t		audit_device_open;
static d_close_t	audit_device_close;
static d_read_t		audit_device_read;
static d_ioctl_t	audit_device_ioctl;
static d_poll_t		audit_device_poll;

static void
audit_wakeup(void);

/*
 * Support for /dev/audit
 */
static struct cdevsw audit_device_cdevsw =
	{ audit_device_open,	audit_device_close,	audit_device_read,	
	  nowrite,		audit_device_ioctl,	nostop,
	  nullreset,		nodevtotty,		audit_device_poll,
	  nommap,		NULL,			"audit",
	  NULL,			-1 };


static struct auditsoftc {
	int	sc_state;		/* see above for possibilities */
	struct	selinfo sc_selp;	/* process waiting on select call */
	struct	sigio *sc_sigio;	/* information for async I/O */
} auditsoftc;


static int	audit_open;
static int	audit_devsw_installed;
#ifdef DEVFS
static void *audit_devfs_token;
#endif /* DEVFS */


/*
 * List to hold buffered records, stored in a form appropriate for immediate
 * copy into userland.
 */
static TAILQ_HEAD(audit_list, audit_list_entry) audit_list_head;
struct audit_list_entry {
	char	*aud_rec_data;		/* data blob */
	int	aud_rec_data_len;	/* length of data blob */

	struct flat_header	fh;	/* the header, seperable */
	int	fh_read_header;		/* the header has been read */

	TAILQ_ENTRY(audit_list_entry)	audit_list_entries;
};

/*
 * Structure to hold a record while under construction in the kernel
 */
struct audit_record {
	uid_t	aud_rec_uid;
	char	*aud_hdr_data, *aud_evinfo_data, *aud_obj_data, *aud_subj_data;
	int	aud_hdr_data_len, aud_evinfo_data_len, aud_obj_data_len,
		aud_subj_data_len;
	int	aud_hdr_data_buflen, aud_evinfo_data_buflen,
		aud_obj_data_buflen, aud_subj_data_buflen;
	int	status_set;
};

MALLOC_DEFINE(M_AUDBUF,
	"auditbuf",
	"Buffers for userland and kernel audit data");

static void
audit_init(void *unused)
{
	dev_t dev;

	/* audit_create(); */
	audit_open = 0;

	if (!audit_devsw_installed) {
		dev = makedev(CDEV_MAJOR, 0);
		cdevsw_add(&dev, &audit_device_cdevsw, NULL);
		audit_devsw_installed = 1;
#ifdef DEVFS
		audit_devfs_token =
	    	devfs_add_devswf(&audit_device_cdevsw, 0, DV_CHR,
		    UID_ROOT, GID_WHEEL, 0400, "audit");
#endif
	}

	TAILQ_INIT(&audit_list_head);
}

SYSINIT(auditdev,SI_SUB_DRIVERS,SI_ORDER_MIDDLE+CDEV_MAJOR,audit_init,NULL)


/*
 * Create a new in-kernel audit event, holding uid responsible, and of type
 * event_type (see sys/audit.h).  Uses M_WAITOK, so should not fail?
 */
void *
new_audit_event(int event_type, uid_t uid)
{
	struct flat_aud_hdr	*f_hdr;
	struct flat_aud_info	*f_info;
	struct audit_record	*rec;
	aud_id_t	*anaud_id_t;
	aud_time_t	*anaud_time_t;
	struct timespec	ts;
	void	*current;
	short	*ashort;
	int	*anint;

	rec = malloc(sizeof(struct audit_record), M_AUDBUF, M_WAITOK);
	if (rec == NULL)
		return(NULL);

	rec->aud_evinfo_data = rec->aud_obj_data = rec->aud_subj_data = NULL;

	rec->aud_evinfo_data_len = rec->aud_obj_data_len =
	    rec->aud_subj_data_len = 0;
	rec->aud_evinfo_data_buflen = rec->aud_obj_data_buflen =
	    rec->aud_subj_data_buflen = 0;

	/* leave room for status, errno to be filled in later */
	rec->aud_hdr_data_len =
	    sizeof(struct flat_aud_hdr) +
	    sizeof(struct flat_aud_info) +
	    sizeof(short) +			/* AUD_FORMAT */
	    sizeof(struct flat_aud_info) +
	    sizeof(short) +			/* AUD_VERSION */
	    sizeof(struct flat_aud_info) +
	    sizeof(aud_id_t) +			/* AUD_AUD_ID */
	    sizeof(struct flat_aud_info) +
	    sizeof(int) +			/* AUD_EVENT_TYPE */
	    sizeof(struct flat_aud_info) +
	    sizeof(aud_time_t);		 	/* AUD_TIME */

	/* here's the status + errno */
	rec->aud_hdr_data_buflen =
	    rec->aud_hdr_data_len +
	    sizeof(struct flat_aud_info) +
	    sizeof(aud_status_t) +		/* AUD_STATUS */
	    sizeof(struct flat_aud_info) +
	    sizeof(int);			/* AUD_ERRNO */

	rec->aud_hdr_data = malloc(rec->aud_hdr_data_buflen, M_AUDBUF,
	    M_WAITOK);
	if (rec->aud_hdr_data == NULL) {
		free(rec, M_AUDBUF);
		return(NULL);
	}

	/*
	 * HDR
	 */
	current = rec->aud_hdr_data;
	f_hdr = current;
	f_hdr->fah_magic = AUD_MAGIC_FHDR;
	f_hdr->fah_num_info = -1;
	current += sizeof(struct flat_aud_hdr);

	/*
	 * AUD_FORMAT
	 */
	f_info = current;
	f_info->fai_magic = AUD_MAGIC_FINFO;
	f_info->fai_type = AUD_TYPE_SHORT;
	f_info->fai_item_id = AUD_FORMAT;
	f_info->fai_length = sizeof(short);
	current += sizeof(struct flat_aud_info);

	ashort = current;
	*ashort = AUD_NATIVE;
	current += sizeof(short);

	/*
	 * AUD_VERSION
	 */
	f_info = current;
	f_info->fai_magic = AUD_MAGIC_FINFO;
	f_info->fai_type = AUD_TYPE_SHORT;
	f_info->fai_item_id = AUD_VERSION;
	f_info->fai_length = sizeof(short);
	current += sizeof(struct flat_aud_info);

	ashort = current;
	*ashort = AUD_STD_NOT;
	current += sizeof(short);

	/*
	 * AUD_AUD_ID
	 */
	f_info = current;
	f_info->fai_magic = AUD_MAGIC_FINFO;
	f_info->fai_type = AUD_TYPE_AUD_ID;
	f_info->fai_item_id = AUD_AUD_ID;
	f_info->fai_length = sizeof(aud_id_t);
	current += sizeof(struct flat_aud_info);

	anaud_id_t = current;
	*anaud_id_t = uid;
	current += sizeof(aud_id_t);

	/*
	 * AUD_EVENT_TYPE
	 */
	f_info = current;
	f_info->fai_magic = AUD_MAGIC_FINFO;
	f_info->fai_type = AUD_TYPE_INT;
	f_info->fai_item_id = AUD_EVENT_TYPE;
	f_info->fai_length = sizeof(int);
	current += sizeof(struct flat_aud_info);

	anint = current;
	*anint = event_type;
	current += sizeof(int);

	/*
	 * AUD_TIME
	 */
	f_info = current;
	f_info->fai_magic = AUD_MAGIC_FINFO;
	f_info->fai_type = AUD_TYPE_AUD_TIME;
	f_info->fai_item_id = AUD_TIME;
	f_info->fai_length = sizeof(aud_time_t);
	current += sizeof(struct flat_aud_info);

	anaud_time_t = current;
	getnanotime(&ts);
	anaud_time_t->sec = ts.tv_sec;
	anaud_time_t->nsec = ts.tv_nsec;
	current += sizeof(aud_time_t);

	rec->status_set = 0;
	rec->aud_rec_uid = uid;

	if (current > rec->aud_hdr_data + rec->aud_hdr_data_len)
		panic("new_aud_event.invariant_failed\n");

	return(rec);
}


/*
 * Put a piece of information into event, section {hdr,evinfo,obj,subj} of
 * type, located at dataptr
 */
int
audit_put(void *event, int section, int item_id, int type, int len,
	  void *dataptr)
{
	struct audit_record	*rec;
	struct flat_aud_info	*f_info;
	struct flat_aud_hdr	*f_hdr;
	struct flat_aud_evinfo	*f_evinfo;
	struct flat_aud_obj	*f_obj;
	struct flat_aud_subj	*f_subj;
	void	*current, *newchunk;
	int	new_len;

	if (!event) {
		printf("audit_put: null event\n");
		return(EINVAL);
	}
	rec = event;

	switch(section) {
	case AUD_MAGIC_HDR:
		new_len = len + sizeof(struct flat_aud_info) +
		    rec->aud_hdr_data_len;
		if (rec->aud_hdr_data_len == 0) {
			/* need space for flat_aud_hdr */
			new_len += sizeof(struct flat_aud_hdr);
		}
		if (new_len > rec->aud_hdr_data_buflen) {
			/* will need more space */
			newchunk = malloc(new_len, M_AUDBUF, M_WAITOK);
			if (newchunk == NULL)
				return(ENOMEM);
			if (rec->aud_hdr_data != NULL) {
				bcopy(rec->aud_hdr_data, newchunk,
				    rec->aud_hdr_data_len);
				free(rec->aud_hdr_data, M_AUDBUF);
			}
			rec->aud_hdr_data = newchunk;
			rec->aud_hdr_data_buflen = new_len;
		}
		if (rec->aud_hdr_data_len == 0) {
			f_hdr = rec->aud_hdr_data;
			f_hdr->fah_magic = AUD_MAGIC_HDR;
			f_hdr->fah_num_info = -1;
			rec->aud_hdr_data_len +=
			    sizeof(struct flat_aud_hdr);
		}
		current = rec->aud_hdr_data + rec->aud_hdr_data_len;
		break;

	case AUD_MAGIC_EVINFO:
		new_len = len + sizeof(struct flat_aud_info) +
		    rec->aud_evinfo_data_len;
		if (rec->aud_evinfo_data_len == 0) {
			/* will need space for a flat_aud_evinfo */
			new_len += sizeof(struct flat_aud_evinfo);
		}
		if (new_len > rec->aud_evinfo_data_buflen) {
			/* will need more space */
			newchunk = malloc(new_len, M_AUDBUF, M_WAITOK);
			if (newchunk == NULL)
				return(ENOMEM);
			if (rec->aud_evinfo_data != NULL) {
				bcopy(rec->aud_evinfo_data, newchunk,
				    rec->aud_evinfo_data_len);
				free(rec->aud_evinfo_data, M_AUDBUF);
			}
			rec->aud_evinfo_data = newchunk;
			rec->aud_evinfo_data_buflen = new_len;
		}
		if (rec->aud_evinfo_data_len == 0) {
			f_evinfo = rec->aud_evinfo_data;
			f_evinfo->fae_magic = AUD_MAGIC_FEVINFO;
			f_evinfo->fae_num_info = -1;
			rec->aud_evinfo_data_len +=
			    sizeof(struct flat_aud_evinfo);
		}
		current = rec->aud_evinfo_data + rec->aud_evinfo_data_len;
		break;

	case AUD_MAGIC_OBJ:
		new_len = len + sizeof(struct flat_aud_info) +
		    rec->aud_obj_data_len;
		if (rec->aud_obj_data_len == 0) {
			/* will need space for a flat_aud_obj */
			new_len += sizeof(struct flat_aud_obj);
		}
		if (new_len > rec->aud_obj_data_buflen) {
			/* will need more space */
			newchunk = malloc(new_len, M_AUDBUF, M_WAITOK);
			if (newchunk == NULL)
				return(ENOMEM);
			if (rec->aud_obj_data != NULL) {
				bcopy(rec->aud_obj_data, newchunk,
				    rec->aud_obj_data_len);
				free(rec->aud_obj_data, M_AUDBUF);
			}
			rec->aud_obj_data = newchunk;
			rec->aud_obj_data_buflen = new_len;
		}
		if (rec->aud_obj_data_len == 0) {
			f_obj = rec->aud_obj_data;
			f_obj->fao_magic = AUD_MAGIC_FOBJ;
			f_obj->fao_num_info = -1;
			rec->aud_obj_data_len +=
			    sizeof(struct flat_aud_obj);
		}
		current = rec->aud_obj_data + rec->aud_obj_data_len;
		break;

	case AUD_MAGIC_SUBJ:
		new_len = len + sizeof(struct flat_aud_info) +
		    rec->aud_subj_data_len;
		if (rec->aud_subj_data_len == 0) {
			/* will need space for a flat_aud_subj */
			new_len += sizeof(struct flat_aud_subj);
		}
		if (new_len > rec->aud_subj_data_buflen) {
			/* will need more space */
			newchunk = malloc(new_len, M_AUDBUF, M_WAITOK);
			if (newchunk == NULL)
				return(ENOMEM);
			if (rec->aud_subj_data) {
				bcopy(rec->aud_subj_data, newchunk,
				    rec->aud_subj_data_len);
				free(rec->aud_subj_data, M_AUDBUF);
			}
			rec->aud_subj_data = newchunk;
			rec->aud_subj_data_buflen = new_len;
		}
		if (rec->aud_subj_data_len == 0) {
			f_subj = rec->aud_subj_data;
			f_subj->fas_magic = AUD_MAGIC_FSUBJ;
			f_subj->fas_num_info = -1;
			rec->aud_subj_data_len +=
			    sizeof(struct flat_aud_subj);
		}
		current = rec->aud_subj_data + rec->aud_subj_data_len;
		break;

	default:
		return(EINVAL);
	}

	/* there is enough room for the flat_aud_info and the data itself */
	f_info = current;
	f_info->fai_magic = AUD_MAGIC_FINFO;
	f_info->fai_type = type;
	f_info->fai_item_id = item_id;

	switch(type) {
	case AUD_TYPE_STRING_ARRAY:
		printf("audit_put: don't know how to deal with arrays yet\n");
		return(EINVAL);
	default:
		f_info->fai_length = len;
		current += sizeof(struct flat_aud_info);
		bcopy(dataptr, current, len);
		current += len;
	}

	switch(section) {
	case AUD_MAGIC_HDR:
		rec->aud_hdr_data_len += sizeof(struct flat_aud_info) + len;
		break;
	case AUD_MAGIC_EVINFO:
		rec->aud_evinfo_data_len += sizeof(struct flat_aud_info) + len;
		break;
	case AUD_MAGIC_OBJ:
		rec->aud_obj_data_len += sizeof(struct flat_aud_info) + len;
		break;
	case AUD_MAGIC_SUBJ:
		rec->aud_subj_data_len += sizeof(struct flat_aud_info) + len;
		break;
	default:
		/* shouldn't get here, as just did this above! */
		panic("unknown section");
	}

	return(0);
}


int
audit_put_subj(void *event, struct proc *p)
{

	return(0);
}


/*
 * Submit a completed audit record for delivery to userland.
 */
int
audit_submit(void *event)
{
	struct audit_list_entry	*ale;
	struct audit_record	*rec;
	char	*current;
	int	error;

	rec = event;

	/* don't create new records while the device is closed */
	if (!audit_open) {
		error = 0;	/* pretend to accept it */
		goto error;
	}

	ale = malloc(sizeof(struct audit_list_entry), M_AUDBUF, M_WAITOK);
	if (ale == NULL) {
		error = ENOMEM;
		goto error;
	}

	ale->aud_rec_data_len = rec->aud_hdr_data_len +
	    rec->aud_evinfo_data_len + rec->aud_obj_data_len +
	    rec->aud_subj_data_len;

	if (!ale->aud_rec_data_len) {
		free(ale, M_AUDBUF);
		error = EINVAL;
		goto error;
	}

	ale->aud_rec_data = malloc(ale->aud_rec_data_len, M_AUDBUF, M_WAITOK);
	if (ale->aud_rec_data == NULL) {
		free(ale, M_AUDBUF);
		error = ENOMEM;
		goto error;
	}

	/* Flattened Record Header */
	ale->fh.fh_magic = AUD_MAGIC_FLAT;
	ale->fh.fh_length = ale->aud_rec_data_len;
	ale->fh.fh_version = AUD_FLAT_VERSION;
	ale->fh_read_header = 0;
	ale->fh.fh_uid = rec->aud_rec_uid;

	current = ale->aud_rec_data;

	if (rec->aud_hdr_data) {
		memcpy(current, rec->aud_hdr_data, rec->aud_hdr_data_len);
		current += rec->aud_hdr_data_len;
	}

	if (rec->aud_evinfo_data) {
		memcpy(current, rec->aud_evinfo_data, rec->aud_evinfo_data_len);
		current += rec->aud_evinfo_data_len;
	}

	if (rec->aud_obj_data) {
		memcpy(current, rec->aud_obj_data, rec->aud_obj_data_len);
		current += rec->aud_obj_data_len;
	}

	if (rec->aud_subj_data) {
		memcpy(current, rec->aud_subj_data, rec->aud_subj_data_len);
		current += rec->aud_subj_data_len;
	}

	if (current > ale->aud_rec_data + ale->aud_rec_data_len)
		panic("audit_submit.invariant failed\n");
	TAILQ_INSERT_TAIL(&audit_list_head, ale, audit_list_entries);
	audit_wakeup();

	error = 0;

error:
	if (rec->aud_hdr_data)
		free(rec->aud_hdr_data, M_AUDBUF);
	if (rec->aud_evinfo_data)
		free(rec->aud_evinfo_data, M_AUDBUF);
	if (rec->aud_obj_data)
		free(rec->aud_obj_data, M_AUDBUF);
	if (rec->aud_subj_data)
		free(rec->aud_subj_data, M_AUDBUF);
	free(rec, M_AUDBUF);

	return(error);
}

static void
audit_flush(void)
{
	while (TAILQ_FIRST(&audit_list_head) != NULL)
		TAILQ_REMOVE(&audit_list_head, audit_list_head.tqh_first,
		    audit_list_entries);
}


static int
audit_device_close(dev_t dev, int flags, int fmt, struct proc *p)
{

	if (minor(dev) != 0)
	    return(ENXIO);

	if (!audit_open)
	    panic("audit_close: not open");

	audit_open = 0;
	auditsoftc.sc_state = 0;
	funsetown(auditsoftc.sc_sigio);

	printf("Auditing log flushed and disabled\n");
	audit_flush();

	return(0);
}

static int
audit_device_open(dev_t dev, int flags, int fmt, struct proc *p)
{

	if (minor(dev) != 0)
	    return(ENXIO);

	if (audit_open) {
	    printf("audit_open: device already open\n");
	    return(EBUSY);
	}

	audit_open = 1;

	fsetown(p->p_pid, &auditsoftc.sc_sigio); /* signal process only */

	/*
	 * reset first record in chain so that it is read for scratch, as
	 * the last listener did not pick up the whole record
	 */

	return(0);
}

static int
audit_device_read(dev_t dev, struct uio *uio, int flags)
{
	struct audit_list_entry	*ale;
	void	*data;
	int	amnt, rv, datalen;
	int	s;

	rv = 0;

	s = splhigh();
	while ((ale = TAILQ_FIRST(&audit_list_head)) == NULL) { 
		if (flags & IO_NDELAY) {
	 		splx(s);
	 		return(EWOULDBLOCK);
	 	}
	  	auditsoftc.sc_state |= AUDIT_RDWAIT;
	  	if ((rv = tsleep((caddr_t) &audit_list_head, AUDIT_RDPRI |
		    PCATCH, "audit", 0))) {
	 		splx(s);
			return(rv);
		}
	}
	splx(s);
	auditsoftc.sc_state &= ~AUDIT_RDWAIT;

	/*
	 * perform copy
	 */
	if (!ale->fh_read_header) {
		data = &(ale->fh);
		datalen = sizeof(struct flat_header);
	} else {
		data = ale->aud_rec_data;
		datalen = ale->aud_rec_data_len;
	}
	while (uio->uio_resid > 0) {
		amnt = MIN(uio->uio_resid, datalen);
		if (amnt == 0)
			break;
		if ((rv = uiomove((caddr_t)data, amnt, uio)) != 0)
			break;
		datalen -= amnt;
	}

	if (datalen != 0) {
		printf("audit_device_read: not all data read in readv()\n");
		/*
		 * junk the remainder of the record if that was just
		 * the flat_header 
	 	 */
		ale->fh_read_header = 1;
	} else {
	}

	if (!ale->fh_read_header) {
		ale->fh_read_header = 1;
	} else {
		TAILQ_REMOVE(&audit_list_head, audit_list_head.tqh_first,
		    audit_list_entries);
		free(ale->aud_rec_data, M_AUDBUF);
		free(ale, M_AUDBUF);
	}
	return(rv);
}

static int
audit_device_ioctl(dev_t dev, u_long cmd, caddr_t data, int flags,
	struct proc *p)
{

	return(ENOTTY);
}

static int
audit_device_poll(dev_t dev, int events, struct proc *p)
{
	int s;
	int revents = 0;

	s = splhigh();

	if (events & (POLLIN | POLLRDNORM))
		if (!TAILQ_EMPTY(&audit_list_head))
			 revents |= events & (POLLIN | POLLRDNORM);
		 else
			selrecord(p, &auditsoftc.sc_selp);

	splx(s);
	return (revents);
}

static void
audit_wakeup(void)
{

	if (!audit_open)
		return;

	selwakeup(&auditsoftc.sc_selp);

	if ((auditsoftc.sc_state & AUDIT_ASYNC) &&
	    (auditsoftc.sc_sigio != NULL))
		pgsigio(auditsoftc.sc_sigio, SIGIO, 0);

	if (auditsoftc.sc_state & AUDIT_RDWAIT) {
		wakeup((caddr_t) &audit_list_head);
		auditsoftc.sc_state &= ~AUDIT_RDWAIT;
	}
}

/*
 * syscall implementations
 */
int
aud_switch_sc(struct proc *p,
	struct aud_switch_sc_args *uap)
{
	aud_state_t	aud_state_in, aud_state_out;
	int	error;

	error = copyin(uap->aud_state, &aud_state_in, sizeof(aud_state_t));
	if (error) {
		if (AUDIT_PROC(p))
			AUDIT_AUD_SWITCH(p, AUD_FAIL_OTHER, error,
			    aud_state_in);
		return(error);
	}

	switch(aud_state_in) {
	case AUD_STATE_OFF:
		error = suser(p->p_ucred, &p->p_acflag);
		if (error) {
			if (AUDIT_PROC(p))
				AUDIT_AUD_SWITCH(p, AUD_FAIL_OTHER, error,
				    aud_state_in);
			return(error);
		}
		p->p_posix1e_flag &= ~POSIX1E_FLAG_AUDIT;
		break;
	case AUD_STATE_ON:
		error = suser(p->p_ucred, &p->p_acflag);
		if (error) {
			if (AUDIT_PROC(p))
				AUDIT_AUD_SWITCH(p, AUD_FAIL_OTHER, error,
				    aud_state_in);
			return(error);
		}
		p->p_posix1e_flag |= POSIX1E_FLAG_AUDIT;
		break;
	case AUD_STATE_QUERY:
		if (p->p_posix1e_flag & POSIX1E_FLAG_AUDIT)
			aud_state_out = AUD_STATE_ON;
		else
			aud_state_out = AUD_STATE_OFF;
		error = copyout(&aud_state_out, uap->aud_state,
		    sizeof(aud_state_t));
		if (error) {
			if (AUDIT_PROC(p))
				AUDIT_AUD_SWITCH(p, AUD_FAIL_OTHER, error,
				    aud_state_in);
			return(error);
		}
		break;
	default:
		if (AUDIT_PROC(p))
			AUDIT_AUD_SWITCH(p, AUD_FAIL_OTHER, EINVAL,
			    aud_state_in);
		return(EINVAL);
	}

	if (AUDIT_PROC(p))
		AUDIT_AUD_SWITCH(p, AUD_SUCCESS, 0, aud_state_in);
	return(0);
}

int
aud_write_sc(struct proc *p,
	     struct aud_write_sc_args *uap)
{
	struct audit_list_entry	*ale;
	char	*buf;
	int	i;

	i = suser(p->p_ucred, &p->p_acflag);
	if (i) {
		AUDIT_AUD_WRITE(p, AUD_FAIL_PRIV, i);
		return(i);
	}

	/* require to be > sizeof the header */
	if (uap->len <= sizeof(struct flat_header)) {
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, EINVAL);
		return(EINVAL);
	}
	if (uap->len > AUDIT_MAX_AUD_WRITE_LEN) {
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, ENOMEM);
		return(ENOMEM);
	}

	ale = (struct audit_list_entry *)
	    malloc(sizeof(struct audit_list_entry),  M_AUDBUF, M_WAITOK);
	if (!ale) {
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, ENOMEM);
		return(ENOMEM);
	}

	/* always at least one byte */
	buf = (char *) malloc(uap->len - sizeof(struct flat_header), M_AUDBUF,
	    M_WAITOK);
	if (!buf) {
		free(ale, M_AUDBUF);
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, ENOMEM);
		return(ENOMEM);
	}

	i = copyin(uap->addr, &(ale->fh), sizeof(struct flat_header));
	if (i) {
		free(ale, M_AUDBUF);
		free(buf, M_AUDBUF);
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, i);
		return(i);
	}

	if (ale->fh.fh_magic != AUD_MAGIC_FLAT) {
		printf("aud_write_sc: bad magic on submitted audit record\n");
		free(ale, M_AUDBUF);
		free(buf, M_AUDBUF);
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, EINVAL);
		return(EINVAL);
	}

	if (ale->fh.fh_length != uap->len - sizeof(struct flat_header)) {
		printf("aud_write_sc: ale->fh.fh_length=%d, uap->len=%d\n",
		    ale->fh.fh_length, uap->len);
		free(ale, M_AUDBUF);
		free(buf, M_AUDBUF);
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, EINVAL);
		return(EINVAL);
	}

	i = copyin(uap->addr + sizeof(struct flat_header), buf, uap->len -
	    sizeof(struct flat_header));
	if (i) {
		free(ale, M_AUDBUF);
		free(buf, M_AUDBUF);
		AUDIT_AUD_WRITE(p, AUD_FAIL_OTHER, i);
		return(i);
	}

	ale->fh.fh_uid = p->p_cred->pc_ucred->cr_uid;
	ale->fh_read_header = 0;
	ale->aud_rec_data_len = uap->len - sizeof(struct flat_header);
	ale->aud_rec_data = buf;

	TAILQ_INSERT_TAIL(&audit_list_head, ale, audit_list_entries);
	audit_wakeup();

	/* don't need to audit success */
	return(0);
}

int
aud_get_id_sc(struct proc *p,
	      struct aud_get_id_sc_args *uap)
{

	return(ENOSYS);
}

#else /* !_POSIX_AUD */

int
aud_switch_sc(struct proc *p,
	      struct aud_switch_sc_args *uap)
{

	return(ENOSYS);
}

int
aud_write_sc(struct proc *p,
	     struct aud_write_sc_args *uap)
{

	return(ENOSYS);
}

int
aud_get_id_sc(struct proc *p,
	      struct aud_get_id_sc_args *uap)
{

	return(ENOSYS);
}

#endif
