/*-
 * Copyright (c) 1999 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *       $Id: $
 */

#ifndef _SYS_AUDIT_H_
#define _SYS_AUDIT_H_
#ifdef _POSIX_AUD

#include <sys/queue.h>

/*
 * Constants required for POSIX.1e Auditing
 */

/* 24-1 Interfaces and Corresponding Audit Events */
#define AUD_AET_FIRST		AUD_AET_NONE
#define AUD_AET_NONE		0	/* none */
#define AUD_AET_AUD_SWITCH	1	/* aud_switch() */
#define AUD_AET_AUD_WRITE	2	/* aud_write() */
#define AUD_AET_CHDIR		3	/* chdir() */
#define AUD_AET_CHMOD		4	/* chmod() */
#define AUD_AET_CHOWN		5	/* chown() */
#define AUD_AET_CREAT		6	/* creat() */
#define AUD_AET_DUP		7	/* dup(), dup2() */
#define AUD_AET_EXEC		8	/* exec(), execl(), execlp(), execv()
					 * execvp(), execle(), execve() */
#define AUD_AET_EXIT		9	/* _exit() */
#define AUD_AET_FORK		10	/* fork() */
#define AUD_AET_KILL		11	/* kill() */
#define AUD_AET_LINK		12	/* link() */
#define AUD_AET_MKDIR		13	/* mkdir() */
#define AUD_AET_MKFIFO		14	/* mkfifo() */
#define AUD_AET_OPEN		15	/* open(), opendir() */
#define AUD_AET_PIPE		16	/* pipe() */
#define AUD_AET_RENAME		17	/* rename() */
#define AUD_AET_RMDIR		18	/* rmdir */
#define AUD_AET_SETGID		19	/* setgid() */
#define AUD_AET_SETUID		20	/* setuid() */
#define AUD_AET_UNLINK		21	/* unlink() */
#define AUD_AET_UTIME		22	/* utime() */

#ifdef _POSIX_ACL
#define AUD_AET_ACL_DELETE_DEF_FILE	22	/* acl_delete_def_file() */
#define AUD_AET_ACL_SET_FD		23	/* acl_set_fd() */
#define AUD_AET_ACL_SET_FILE		24	/* acl_set_file() */
#endif

#ifdef _POSIX_CAP
#define AUD_AET_CAP_SET_FD		25	/* cap_set_fd() */
#define AUD_AET_CAP_SET_FILE		26	/* cap_set_file() */
#define AUD_AET_CAP_SET_PROC		27	/* cap_set_proc() */
#endif

#ifdef _POSIX_INF
#define AUD_AET_INF_SET_FD		28	/* inf_set_fd() */
#define AUD_AET_INF_SET_FILE		29	/* inf_set_file() */
#define AUD_AET_INF_SET_PROC		30	/* inf_set_proc() */
#endif

#ifdef _POSIX_MAC
#define AUD_AET_MAC_SET_FD		31	/* mac_set_fd() */
#define AUD_AET_MAC_SET_FILE		32	/* mac_set_file() */
#define AUD_AET_MAC_SET_PROC		33	/* mac_set_proc() */
#endif

/* BSD-only events */
#define AUD_AET_BSD_MISC		34	/* bsd_misc()... */
#define AUD_AET_LAST			AUD_AET_BSD_MISC


/*
 * Table 24-5 Values for the aud_info_type_t Member
 * NOTE: negative values are reserved for the implementation
 */
#define AUD_TYPE_ACL		1	/* acl_t* */
#define AUD_TYPE_ACL_TYPE	2	/* acl_type_t* */
#define AUD_TYPE_AUD_ID		3	/* aud_id_t* */
#define AUD_TYPE_AUD_OBJ_TYPE	4	/* aud_obj_type_t* */
#define AUD_TYPE_AUD_STATE	5	/* aud_state_t* */
#define AUD_TYPE_AUD_STATUS	6	/* aud_status_t* */
#define AUD_TYPE_AUD_TIME	7	/* aud_time_t* */
#define AUD_TYPE_CAP		8	/* cap_t* */
#define AUD_TYPE_CHAR		9	/* char* */
#define AUD_TYPE_GID		10	/* gid_t* */
#define AUD_TYPE_INF		11	/* inf_t* */
#define AUD_TYPE_INT		12	/* int* */
#define AUD_TYPE_LONG		13	/* long* */
#define AUD_TYPE_MAC		14	/* mac_t* */
#define AUD_TYPE_MODE		15	/* mode_t* */
#define AUD_TYPE_OPAQUE		16	/* void* */
#define AUD_TYPE_PID		17	/* pid_t* */
#define AUD_TYPE_SHORT		18	/* short* */
#define AUD_TYPE_STRING		19	/* char*, pointed to a null terminated
					 * character string */
#define AUD_TYPE_STRING_ARRAY	20	/* char** */
#define AUD_TYPE_TIME		21	/* time_t* */
#define AUD_TYPE_UID		22	/* uid_t* */

/* internal values used by auditing code */
#define AUD_TYPE_IMP_AUD_REC	-64	/* aud_rec_t */
#define AUD_TYPE_IMP_EVID	-65	/* list of events */


/* Table 24-6 aud_obj_type_t Values */
#define AUD_OBJ_BLOCK_DEV	1	/* Block devices */
#define AUD_OBJ_CHAR_DEV	2	/* Character devices */
#define AUD_OBJ_DIR		3	/* Directory */
#define AUD_OBJ_FIFO		4	/* FIFO object */
#define AUD_OBJ_FILE		5	/* Regular file */
#define AUD_OBJ_PROC		6	/* Process object */


/* Table 24-7 aud_state_t Values */
#define AUD_STATE_OFF		1	/* System events not audited */
#define AUD_STATE_ON		2	/* System events audited */
#define AUD_STATE_QUERY		3	/* Enquiry value for aud_switch() */


/* Table 24-8 aud_status_t Values */
#define AUD_FAIL_PRIV		1	/* The event failed because the
					 * process did not have appropriate
					 * privilege. */
#define AUD_FAIL_DAC		2	/* The event failed because of DAC
					 * access checks. */
#define AUD_FAIL_MAC		3	/* The event failed because of MAC
					 * access checks. */
#define AUD_FAIL_OTHER		4	/* The event failed for some reason
					 * not included in other AUD_FAIL_*
					 * values. */
#define AUD_PRIV_USED		5	/* The event completed successfully;
					 * appropriate privilege was used. */
#define AUD_SUCCESS		6	/* The event completed successfully. */


/*
 * Combined:
 *   Table 24-10 aud_hdr_info_p Values
 *   Table 24-11 aud_obj_info_p Values
 *   Table 24-12 aud_subj_info_p Values
 *   Section 24.2.2 aud_evinfo_info_p values
 */
#define AUD_FIRST_ITEM		-1	/* First of items in header/etc */
#define AUD_NEXT_ITEM		-2	/* Next item in header/etc */
#define AUD_LAST_ITEM		-3	/* Last item in header/etc */
#define AUD_FORMAT		1	/* The format of the audit record;
					 * AUD_TYPE_SHORT */
#define AUD_VERSION		2	/* The version of the record;
					 * AUD_TYPE_SHORT */
#define AUD_AUD_ID		3	/* The audit ID of the process;
					 * AUD_TYPE_AUD_ID */
#define AUD_EVENT_TYPE		4	/* The event type of this record;
					 * AUD_TYPE_INT or AUD_TYPE_STRING */
#define AUD_TIME		5	/* The time the event occurred;
					 * AUD_TYPE_AUD_TIME */
#define AUD_STATUS		6	/* The audit status of the event;
					 * AUD_TYPE_AUD_STATUS */
#define AUD_ERRNO		7	/* Value returned for the event (errno)
					 * AUD_TYPE_INT */
#define AUD_TYPE		8	/* The type of the object;
					 * AUD_TYPE_AUD_OBJ_TYPE */
#define AUD_UID			9	/* The user ID of the object owner;
					 * AUD_TYPE_UID */
#define AUD_GID			10	/* The group ID of the object owner;
					 * AUD_TYPE_GID */
#define AUD_MODE		11	/* The mode bits of the object;
					 * AUD_TYPE_MODE */
#define AUD_NAME		12	/* The name of the object;
					 * AUD_TYPE_STRING */
#define AUD_ACL			13	/* The ACL of the object;
					 * AUD_TYPE_ACL */
#define AUD_MAC_LBL		14	/* The MAC label of the
					 * object/process;
					 * AUD_TYPE_MAC */
#define AUD_INF_LBL		15	/* The information label of the
					 * object/process;
					 * AUD_TYPE_INF */
#define AUD_CAP			16	/* The capability set of the
					 * object/process;
					 * AUD_TYPE_CAP */
#define AUD_PID			17	/* The process ID;
					 * AUD_TYPE_PID */
#define AUD_EUID		18	/* The effective user ID;
					 * AUD_TYPE_UID */
#define AUD_EGID		19	/* the effective group ID;
					 * AUD_TYPE_GID */
#define AUD_SGIDS		20	/* The supplementary group IDs;
					 * AUD_TYPE_GID */
#define AUD_RUID		21	/* The real user ID;
					 * AUD_TYPE_UID */
#define AUD_RGID		22	/* The real group ID;
					 * AUD_TYPE_GID */
#define AUD_PATHNAME		231	/* Pathname;
					 * AUD_TYPE_STRING */
#define AUD_FILE_ID		2	/* File Descriptor;
					 * AUD_TYPE_INT */
#define AUD_ACL_TYPE		3	/* ACL Type;
					 * AUD_TYPE_ACL_TYPE */
#define AUD_AUDIT_STATE		5	/* Audit State;
					 * AUD_TYPE_AUDIT_STATE */
#define AUD_RETURN_ID		10	/* Returned PID/FD/etc;
					 * AUD_TYPE_INT */
#define AUD_CMD_ARGS		11	/* Command Arguments;
					 * AUD_TYPE_STRING_ARRAY */
#define AUD_ENVP		12	/* Environmental Arguments;
					 * AUD_TYPE_STRING_ARRAY */
#define AUD_UID_ID		13	/* Effective UID;
					 * AUD_TYPE_UID */
#define AUD_GID_ID		14	/* Effective GID;
					 * AUD_TYPE_GID */
#define AUD_EXIT_CODE		15	/* Process exit code;
					 * AUD_TYPE_INT */
#define AUD_SIG			17	/* Signal Number;
					 * AUD_TYPE_INT */
#define AUD_LINKNAME		19	/* Linkname;
					 * AUD_TYPE_STRING */
#define AUD_OFLAG		21	/* Open Flags;
					 * AUD_TYPE_INT */
#define AUD_RD_FILE_ID		22	/* Read File ID;
					 * AUD_TYPE_INT */
#define AUD_WR_FILE_ID		23	/* Write File ID;
					 * AUD_TYPE_INT */
#define AUD_OLD_PATHNAME	24	/* Old Pathname;
					 * AUD_TYPE_STRING */
#define AUD_NEW_PATHNAME	25	/* New Pathname;
					 * AUD_TYPE_STRING */
#define AUD_ATIME		26	/* Access time;
					 * AUD_TYPE_TIME */
#define AUD_MTIME		27	/* Modification time;
					 * AUD_TYPE_TIME */


/* Table 24.3 Other Constants */
#define AUD_SYSTEM_LOG	-2	/* Value of the filedes argument for
				 * aud_write() */
#define AUD_NATIVE	28	/* Value of the format item in a record
				 * header */
/* If it ever happens:
 * #define AUD_STD_NNNN_N	 * Value of the version item in a record
 *				 * header. *
 */
#define AUD_STD_NOT		4729

/*
 * Magic values to do probabilistic runtime type helping
 */
#define AUD_MAGIC_EVINFO	0x1e69fa72
#define AUD_MAGIC_HDR		0xf0436966
#define AUD_MAGIC_OBJ		0x734463d2
#define AUD_MAGIC_SUBJ		0x7f04a84c
#define AUD_MAGIC_AUDREC	0xb5f77832
#define AUD_MAGIC_FLAT		0xbd6e317c

#define AUD_MAGIC_FEVINFO	0x81445135
#define AUD_MAGIC_FHDR		0x18c65445
#define AUD_MAGIC_FOBJ		0xa2060308
#define AUD_MAGIC_FSUBJ		0x0bed22f3
#define AUD_MAGIC_FINFO		0x61d8f436

/* Table 24-2 Audit Data Types */

typedef void	*aud_descriptor;
typedef int	aud_state_t;
typedef int	aud_status_t;
typedef int	aud_obj_type_t;
typedef uid_t	aud_id_t;
typedef void	*aud_rec_t;
typedef aud_descriptor	aud_evinfo_t;
typedef aud_descriptor	aud_hdr_t;
typedef aud_descriptor	aud_obj_t;
typedef aud_descriptor	aud_subj_t;

LIST_HEAD(aud_info_list, aud_info_t);		/* aud_info_head */
struct aud_info_t {
	int	magic;			/* not portable - XXXX not used */

	int	aud_info_type;		/* standard field */
	size_t	aud_info_length;	/* standard field */
	void *	aud_info_p;		/* standard field */

	/* These fields are not portable and should not accessed directly */
	int	item_id;
	LIST_ENTRY(aud_info_t)		aud_infos;
};
typedef struct aud_info_t	aud_info_t;

struct aud_time_t {
	time_t	sec;	/* Seconds */
	long	nsec;	/* Nanoseconds */
};
typedef struct aud_time_t	aud_time_t;

/*
 * Internal structures used by the audit library; do not assume these will
 * remain the same; instead, use the access functions described in POSIX.1e
 * and in posix_aud(3).  These are the structures eventually accessed via
 * the aud_descriptors above.
 */

struct aud_evinfo_str {
	/* These fields are not portable */
	int	magic;

	/* the record this evinfo is associated with, if any */
	struct aud_rec_str	*myrec;

	/* evinfos contain info fields */
	int	aud_info_num;
	aud_info_t	*aud_info_cur;
	struct aud_info_list	aud_info_head;

	/* events may be in lists of events */
	LIST_ENTRY(aud_evinfo_str)	aud_evinfos;
};

struct aud_hdr_str {
	/* These fields are not portable */
	int	magic;

	/* the record this header is associated with, if any */
	struct aud_rec_str	*myrec;

	/* headers contain info fields */
	int	aud_info_num;
	aud_info_t	*aud_info_cur;
	struct aud_info_list	aud_info_head;

	/* headers may be in lists of headers */
	LIST_ENTRY(aud_hdr_str)	aud_hdrs;
};

struct aud_obj_str {
	/* These fields are not portable */
	int	magic;

	/* the record this object is associated with, if any */
	struct aud_rec_str	*myrec;

	/* objects contain info fields */
	int	aud_info_num;
	aud_info_t	*aud_info_cur;
	struct aud_info_list	aud_info_head;

	/* objects may be in lists of objects */
	LIST_ENTRY(aud_obj_str)	aud_objs;
};

struct aud_subj_str {
	/* These fields are not portable */
	int	magic;

	/* the record this subject is associated with, if any */
	struct aud_rec_str	*myrec;

	/* subjects contain info fields */
	int	aud_info_num;
	aud_info_t	*aud_info_cur;
	struct aud_info_list	aud_info_head;

	/* subjects may be in lists of subjects */
	LIST_ENTRY(aud_subj_str)	aud_subjs;
};

LIST_HEAD(aud_hdr_list, aud_hdr_str);		/* aud_hdr_head */
LIST_HEAD(aud_subj_list, aud_subj_str);		/* aud_subj_head */
LIST_HEAD(aud_evinfo_list, aud_evinfo_str);	/* aud_evinfo_head */
LIST_HEAD(aud_obj_list, aud_obj_str);		/* aud_obj_head */

struct aud_rec_str {
	/* These fields are not portable */
	int	magic;
	/* one or more headers */
	int	aud_num_hdr;
	struct aud_hdr_list	aud_hdr_head;

	/* one or more subject attributes */
	int	aud_num_subj;
	struct aud_subj_list	aud_subj_head;

	/* zero or more sets of events-specific data */
	int	aud_num_evinfo;
	struct aud_evinfo_list	aud_evinfo_head;

	/* zero or more sets of object attributes */
	int	aud_num_obj;
	struct aud_obj_list	aud_obj_head;
};

/*
 * These are flattened versions of the structures used for byte-storage
 * based manipulation, both between kernel/auditd, and for file storage
 * These should never be directly manipulated, instead use the access
 * functions to convert these to aud_rec_t
 */

#define AUD_FLAT_VERSION	1	/* version of this representation */

struct flat_header {
        int     fh_magic;
        int     fh_length;
        int     fh_version;
	uid_t	fh_uid;
};

 /*
struct flat_aud_rec {
        int     far_num_hdr;
        int     far_num_subj;
        int     far_num_evinfo;
        int     far_num_obj;
};
*/
 
struct flat_aud_hdr {
	int	fah_magic;
	int	fah_num_info;
};
 
struct flat_aud_subj {
	int	fas_magic;
	int	fas_num_info;
};
 
struct flat_aud_evinfo {
	int	fae_magic;
	int	fae_num_info;
};
 
struct flat_aud_obj {
	int	fao_magic;
	int	fao_num_info;
};
 
struct flat_aud_info {
	int	fai_magic;
        int     fai_type;
	int	fai_item_id;
        size_t  fai_length;
};


/*
 * These are the access functions described in POSIX.1E
 */

#ifndef KERNEL
/* 24.4 Functions */
ssize_t	aud_copy_ext(void *aud_rec_ext_p, aud_rec_t aud_rec_int,
		     ssize_t size);
aud_rec_t	aud_copy_int(const void *aud_rec_ext_p);
int	aud_delete_event(aud_evinfo_t aud_event_d);
int	aud_delete_event_info(aud_evinfo_t au_event_d, int item_id);
int	aud_delete_hdr(aud_hdr_t aud_hdr_d);
int	aud_delete_hdr_info(aud_hdr_t aud_hdr_d, int item_id);
int	aud_delete_obj(aud_obj_t aud_obj_d);
int	aud_delete_obj_info(aud_obj_t aud_obj_d, int item_id);
int	aud_delete_subj(aud_subj_t aud_subj_d);
int	aud_delete_subj_info(aud_subj_t aud_subj_d, int item_id);
aud_rec_t	aud_dup_record(aud_rec_t ar);
int	aud_evid_from_text(const char *text);
char	*aud_evid_to_text(int event_type, ssize_t *aud_info_length);
int	aud_free(void *obj_p);
int	*aud_get_all_evid(void);
int	aud_get_event(aud_rec_t ar, int index, aud_evinfo_t *aud_event_p);
int	aud_get_event_info(aud_evinfo_t aud_event_d, int item_id,
			   aud_info_t *aud_event_info_p);
int	aud_get_hdr(aud_rec_t ar, int index, aud_hdr_t *aud_hdr_p);
int	aud_get_hdr_info(aud_hdr_t aud_hdr_d, int item_id,
	aud_info_t *aud_hdr_info_p);
aud_id_t	aud_get_id(pid_t pid);
int	aud_get_obj(aud_rec_t ar, int index, aud_obj_t *aud_obj_p);
int	aud_get_obj_info(aud_obj_t aud_obj_d, int item_id,
	aud_info_t *aud_obj_info_p);
int	aud_get_subj(aud_rec_t ar, int index, aud_subj_t *aud_subj_p);
int	aud_get_subj_info(aud_subj_t aud_subj_d, int item_id,
	aud_info_t *aud_subj_info_p);
aud_id_t	aud_id_from_text(const char *text_p);
char	*aud_id_to_text(aud_id_t audit_ID, ssize_t *len_p);
aud_rec_t	aud_init_record(void);
int	aud_put_event(aud_rec_t ar, const aud_evinfo_t *next_p,
		      aud_evinfo_t *new_p);
int	aud_put_event_info(aud_evinfo_t aud_event_d, int position, int item_id,
			   const aud_info_t *aud_event_info_p);
int	aud_put_hdr(aud_rec_t ar, const aud_hdr_t *next_p,
		    aud_hdr_t *new_p);
int	aud_put_hdr_info(aud_hdr_t aud_hdr_d, int position, int item_id,
			 const aud_info_t *aud_hdr_info_p);
int	aud_put_obj(aud_rec_t ar, const aud_obj_t *next_p, aud_obj_t *new_p);
int	aud_put_obj_info(aud_obj_t aud_obj_d, int position, int item_id,
			 const aud_info_t *aud_obj_info_p);
int	aud_put_subj(aud_rec_t ar, const aud_subj_t *next_p,
		     aud_subj_t *new_p);
int	aud_put_subj_info(aud_subj_t aud_subj_d, int position, int item_id,
			  const aud_info_t *aud_subj_info_p);
aud_rec_t	aud_read(int filedes);
char	*aud_rec_to_text(aud_rec_t ar, ssize_t *len_p);
ssize_t	aud_size(aud_rec_t ar);
aud_state_t	aud_switch(aud_state_t aud_state);
int	aud_valid(aud_rec_t ar);
int	aud_write(int filedes, aud_rec_t ar);

/* NOT portable */
aud_rec_t	aud_read_sc(int filedes, uid_t *uid);

#else /* KERNEL */

MALLOC_DECLARE(M_AUDBUF);
#define	POSIX1E_FLAG_AUDIT	0x00000001

void	*new_audit_event(int event_type, uid_t uid);
int	audit_put(void *event, int section, int item_id, int type, int len,
		  void *dataptr);
int	audit_put_subj(void *event, struct proc *p);
int	audit_submit(void *event);

#define AUDIT_PROC(proc) ((proc)->p_posix1e_flag & POSIX1E_FLAG_AUDIT)

#define AUDIT_EXIT(proc, aud_status, exitcode)				\
{									\
	void	*MACRO_audit_event;					\
	int	MACRO_audit_int;					\
	aud_status_t	data_aud_status_t = aud_status;			\
	int	data_int = exitcode, data_int_2 = 0;			\
	MACRO_audit_event = new_audit_event(AUD_AET_EXIT,		\
	    proc->p_cred->pc_ucred->cr_uid);				\
	if (MACRO_audit_event) {					\
		MACRO_audit_int = audit_put_subj(MACRO_audit_event,	\
		    proc);						\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_EVINFO, AUD_EXIT_CODE, AUD_TYPE_INT,	\
		    sizeof(int), &data_int);				\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_STATUS, AUD_TYPE_AUD_STATUS,	\
		    sizeof(aud_status_t), &data_aud_status_t);		\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_ERRNO, AUD_TYPE_INT, 		\
		    sizeof(int), &data_int_2);				\
		MACRO_audit_int += audit_submit(MACRO_audit_event);	\
		if (MACRO_audit_int)					\
			printf("AUDIT_EXIT: generated incomplete "	\
			    "record\n");				\
	} else								\
		printf("AUDIT_EXIT: unable to allocate event\n");	\
}

#define AUDIT_AUD_SWITCH(proc, aud_status, error, aud_state)		\
{									\
	void	*MACRO_audit_event;					\
	int	MACRO_audit_int;					\
	aud_state_t	data_aud_state_t = aud_state;			\
	aud_status_t	data_aud_status_t = aud_status;			\
	int	data_int = error;					\
	MACRO_audit_event = new_audit_event(AUD_AET_AUD_SWITCH,		\
	    proc->p_cred->pc_ucred->cr_uid);				\
	if (MACRO_audit_event) {					\
		MACRO_audit_int = audit_put_subj(MACRO_audit_event,	\
		    proc);						\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_EVINFO, AUD_AUDIT_STATE,			\
		    AUD_TYPE_AUD_STATE, sizeof(aud_state_t),		\
		    &data_aud_state_t);					\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_STATUS, AUD_TYPE_AUD_STATUS,	\
		    sizeof(aud_status_t), &data_aud_status_t);		\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_ERRNO, AUD_TYPE_INT,		\
		    sizeof(int), &data_int);				\
		MACRO_audit_int += audit_submit(MACRO_audit_event);	\
		if (MACRO_audit_int)					\
			printf("AUDIT_AUD_SWITCH: generated "		\
			    "incomplete record\n");			\
	} else								\
		printf("AUDIT_AUD_SWITCH: unable to allocate event\n");	\
}

#define AUDIT_AUD_WRITE(proc, aud_status, error)			\
{									\
	void	*MACRO_audit_event;					\
	int	MACRO_audit_int;					\
	aud_status_t	data_aud_status_t = aud_status;			\
	int	data_int = error;					\
	MACRO_audit_event = new_audit_event(AUD_AET_AUD_WRITE,		\
	    proc->p_cred->pc_ucred->cr_uid);				\
	if (MACRO_audit_event) {					\
		MACRO_audit_int = audit_put_subj(MACRO_audit_event,	\
		    proc);						\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_STATUS, AUD_TYPE_AUD_STATUS,	\
		    sizeof(aud_status_t), &data_aud_status_t);		\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_ERRNO, AUD_TYPE_INT,		\
		    sizeof(int), &data_int);				\
		MACRO_audit_int += audit_submit(MACRO_audit_event);	\
		if (MACRO_audit_int)					\
			printf("AUDIT_AUD_SWITCH: generated "		\
			    "incomplete record\n");			\
	} else								\
		printf("AUDIT_AUD_WRITE: unable to allocate event\n");	\
}

#define AUDIT_CHDIR(proc, aud_status, error, pathname)			\
{									\
	void	*MACRO_audit_event;					\
	int	MACRO_audit_int;					\
	aud_status_t	data_aud_status_t = aud_status;			\
	int	data_int = error;					\
	MACRO_audit_event = new_audit_event(AUD_AET_CHDIR,		\
	    proc->p_cred->pc_ucred->cr_uid);				\
	if (MACRO_audit_event) {					\
		MACRO_audit_int = audit_put_subj(MACRO_audit_event,	\
		    proc);						\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_EVINFO, AUD_PATHNAME, AUD_TYPE_STRING,	\
		    strlen(pathname)+1, pathname);			\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_STATUS, AUD_TYPE_AUD_STATUS,	\
		    sizeof(aud_status_t), &data_aud_status_t);		\
		MACRO_audit_int += audit_put(MACRO_audit_event,		\
		    AUD_MAGIC_HDR, AUD_ERRNO, AUD_TYPE_INT,		\
		    sizeof(int), &data_int);				\
		MACRO_audit_int += audit_submit(MACRO_audit_event);	\
		if (MACRO_audit_int)					\
			printf("AUDIT_CHDIR: generated incomplete "	\
			    "record\n");				\
	} else								\
		printf("AUDIT_CHDIR: unable to allocate event\n");	\
}

#endif /* KERNEL */

#else /* _POSIX_AUD */

#ifdef KERNEL
#define AUDIT_PROC(proc)	(0)
#define AUDIT_AUD_SWITCH(proc, aud_status, error, aud_state)	{}
#define AUDIT_AUD_WRITE(proc, aud_status, error)	{}

#define AUDIT_CHDIR(proc, aud_status, error, pathname)	{}
#define AUDIT_EXIT(proc, aud_status, exitcode)	{}

#endif /* KERNEL */

#endif /* _POSIX_AUD */
#endif /* _SYS_AUDIT_H_ */
