/*-
 * Copyright (c) 1999 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *       $Id: $
 */

#include <sys/types.h>
#include <sys/audit.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define PATH_1	"/etc/motd"
#define PATH_2	"/tmp/motd.ed"
#define PATH_3	"/usr/local/bin/ssh"

char *
info_display(aud_info_t *info)
{
	static char	out[1025];

	out[1024] = '\0';

	switch(info->aud_info_type) {
	case AUD_TYPE_STRING:
		strncpy(out, info->aud_info_p, 1024);
		break;
	case AUD_TYPE_INT:
		snprintf(out, 1024, "%d", *(int *)info->aud_info_p);
		break;
	default:
		sprintf(out, "-undisplayable-");
	}

	return(out);
}


void display_record(FILE *fp, aud_rec_t rec) {
	aud_evinfo_t	evinfo;
	aud_hdr_t	hdr;
	aud_obj_t	obj;
	aud_subj_t	subj;
	aud_info_t	info;
	int	num_evinfo, num_hdr, num_obj, num_subj;
	int	i, count;
	
	num_evinfo = aud_get_event(rec, 1, NULL);
	if (num_evinfo == -1) {
		perror("aud_get_event");
		return;
	}

	num_hdr = aud_get_hdr(rec, 1, NULL);
	if (num_hdr == -1) {
		perror("aud_get_hdr");
		return;
	}

	num_obj = aud_get_obj(rec, 1, NULL);
	if (num_obj == -1) {
		perror("aud_get_obj");
		return;
	}

	num_subj = aud_get_subj(rec, 1, NULL);
	if (num_subj == -1) {
		perror("aud_get_subj");
		return;
	}

	fprintf(fp, "%d events, %d headers, %d objects, %d subjects\n",
	    num_evinfo, num_hdr, num_obj, num_subj);

	for (count = 1; count <= num_evinfo; count++) {
		i = aud_get_event(rec, count, &evinfo);
		if (i == -1) {
			perror("aud_get_event_info");
			return;
		}
		i = aud_get_event_info(evinfo, 0, NULL);
		if (i == -1) {
			perror("aud_get_event_info");
			return;
		}
		fprintf(fp, "record has %d info fields\n", i);
	}
}

void main(void) {
	aud_rec_t	rec_in, rec_out, rec_dup;
	aud_evinfo_t	evinfo_in, evinfo_out;
	aud_obj_t	obj_in, obj_out;
	aud_info_t	info_in, info_out;
	int	i;

	char	*buf;
	int	buflen, reallen;


	rec_in = aud_init_record();

	if (!rec_in) {
		perror("aud_init_record");
		exit(-1);
	}

	fprintf(stdout, "aud_init_record: resulting record size %d\n",
	    aud_size(rec_in));

	i = aud_put_event(rec_in, NULL, &evinfo_in);
	if (i == -1) {
		perror("aud_put_event");
		exit(-1);
	}

	fprintf(stdout, "aud_put_event: resulting record size %d\n",
	    aud_size(rec_in));

	info_in.aud_info_type = AUD_TYPE_STRING;
	info_in.aud_info_length = strlen(PATH_1) + 1;
	info_in.aud_info_p = PATH_1;

	i = aud_put_event_info(evinfo_in, AUD_LAST_ITEM, AUD_RD_FILE_ID,
	    &info_in);
	if (i == -1) {
		perror("aud_put_event_info");
		exit(-1);
	}

	fprintf(stdout, "aud_put_event_info: resulting record size %d\n",
	    aud_size(rec_in));

	info_in.aud_info_type = AUD_TYPE_STRING;
	info_in.aud_info_length = strlen(PATH_2) + 1;
	info_in.aud_info_p = PATH_2;

	i = aud_put_event_info(evinfo_in, AUD_LAST_ITEM, AUD_WR_FILE_ID,
	    &info_in);
	if (i == -1) {
		perror("aud_put_event_info");
		exit(-1);
	}

	fprintf(stdout, "aud_put_event_info: resulting record size %d\n",
	    aud_size(rec_in));

	i = aud_put_obj(rec_in, NULL, &obj_in);
	if (i == -1) {
		perror("aud_put_obj");
		exit(-1);
	}

	info_in.aud_info_type = AUD_TYPE_STRING;
	info_in.aud_info_length = strlen(PATH_3) + 1;
	info_in.aud_info_p = PATH_3;

	i = aud_put_obj_info(obj_in, AUD_LAST_ITEM, AUD_PATHNAME,
	    &info_in);
	if (i == -1) {
		perror("aud_put_obj_info");
		exit(-1);
	}

	fprintf(stdout, "aud_put_obj_info: resulting record size %d\n",
	    aud_size(rec_in));

	/*
	 * Now retrieve the event
	 */

	i = aud_get_event(rec_in, 1, &evinfo_out);
	if (i == -1) {
		perror("aud_get_event");
		exit(-1);
	}

	i = aud_get_event_info(evinfo_out, AUD_FIRST_ITEM, &info_out);
	if (i == -1) {
		perror("aud_get_event_info");
		exit(-1);
	}

	i = aud_get_event_info(evinfo_out, AUD_NEXT_ITEM, &info_out);
	if (i == -1) {
		perror("aud_get_event_info");
		exit(-1);
	}

	fprintf(stdout, "----- Externalize and Internalize Tests -----\n");
	fprintf(stdout, "old record: \n");
	display_record(stdout, rec_in);

	/*
	 * Convert the record to bytes, and back
	 */
	buflen = aud_size(rec_in);
	buf = (char *) malloc(buflen);
	if (buf == NULL) {
		perror("malloc");
		exit(-1);
	}

	reallen = aud_copy_ext(buf, rec_in, buflen);
	if (reallen == -1) {
		perror("aud_copy_ext");
		exit(-1);
	}
	
	fprintf(stdout, "aud_copy_ext returned %d bytes\n", reallen);

	rec_out = aud_copy_int(buf);
	if (rec_out == NULL) {
		perror("aud_copy_int");
		exit(-1);
	}

	fprintf(stdout, "new record: \n");
	display_record(stdout, rec_out);

	/* 
	 * dup the record 
	 */
	rec_dup = aud_dup_record(rec_in);
	if (rec_dup == NULL) {
		perror("aud_dup_record");
		exit(-1);
	}

	fprintf(stdout, "dup record: \n");
	display_record(stdout, rec_dup);
	
}
