/*-
 * Copyright (c) 1999 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *       $Id: posix_mac.c,v 1.4 1999/06/23 12:10:35 ilmar Exp $
 */

/*
 * Supporting user-land library code for POSIX.1e mandatory access controls
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/queue.h>
#include <sys/mac.h>
#include <sys/audit.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/proc.h>
#include <sys/stat.h>

mac mac_max( mac_t a, mac_t b)
{ return ( ((mac)*a > (mac)*b) ? *a : *b); }

mac mac_min( mac_t a, mac_t b)
{ return ( ((mac)*a < (mac)*b) ? *a : *b); }

void mac_enc(mac_t label)
{
	*label = (mac)((int)*label + (int)1);
}

void mac_dec(mac_t label)
{
	*label = (mac)((int)*label - (int)1);
}

int	mac_dominate(mac_t labela, mac_t labelb) // libposix1e
{
	if(!mac_valid(labela) || !mac_valid(labelb)) {
		errno = EINVAL; /* Invalid argument */
		return (-1);
	} else 
		return (labela >= labelb);
		/* 1 - if labela dominates labelb */
		/* 0 - otherwise */
}

int	mac_equal(mac_t labela, mac_t labelb) // libposix1e
{
	if(!mac_valid(labela) || !mac_valid(labelb)) {
		errno = EINVAL; /* Invalid argument */
		return (-1);
	} else 
		return (labela == labelb);
		/* 1 - if labela equals labelb */
		/* 0 - otherwise */
}

int	mac_free(void *buf_p) // libposix1e
{
	if(buf_p) { 
		free(buf_p);
		return (0);
	} else {
		errno = EFAULT; /* Bad address */
		return (-1);
	}
}

mac_t	mac_from_text(const char *text_p) // libposix1e
{
	mac_t	label;

	label = malloc(mac_size());
	if(label == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	*label = (mac)atoi(text_p);
	if(!mac_valid(label)) {
		errno = EINVAL;
		return (NULL);
	}
	
	return (label);
}

mac_t	mac_get_fd(int fildes) // kernel
{
	int error;
	mac_t	flabel;
	struct stat sb;

	if((flabel = malloc(mac_size())) == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	error = mac_get_fd_sc(fildes, flabel);

	/* XXX: I think that error and errno will be setup in kernel */

	if(error) {
		perror("while getting descriptors' mac label");
		mac_free(flabel);
		return (NULL);
	} else 
		return (flabel);
}

mac_t	mac_get_file(const char *path_p) // kernel
{
	int error;
	mac_t	flabel;

	if((flabel = malloc(mac_size())) == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	error = mac_get_file_sc(path_p, flabel);
	if(error) {
		perror("while getting files' mac label");
		mac_free(flabel);
		return (NULL);
	} else {
		return (flabel);
	}
}

mac_t	mac_get_proc(void) // kernel
{
	int error;
	mac_t plabel;

	if((plabel = malloc(mac_size())) == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	error = mac_get_proc_sc(plabel);
	if(error) {
		perror("while getting proc mac label");
		mac_free(plabel);
		return (NULL);
	} else
		return (plabel);
}

mac_t	mac_glb(mac_t labela, mac_t labelb) // ??? libposix1e
{
	mac_t label;

	if(!mac_valid(labela) || !mac_valid(labelb)) {
		errno = EINVAL; /* Invalid argument */
		return (NULL);
	}

	if((label = malloc(mac_size())) == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	*label == mac_min(labela,labelb);
	mac_dec(label); /* decreasing */

	if(!mac_valid(label)) {
		errno = ENOENT;
		mac_free(label);
		return (NULL);
	} else
		return (label);
}

mac_t	mac_lub(mac_t labela, mac_t labelb) // ??? libposix1e
{
	mac_t label;

	if(!mac_valid(labela) || !mac_valid(labelb)) {
		errno = EINVAL;
		return (NULL);
	}
	
	if((label = malloc(mac_size())) == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	*label = mac_max(labela,labelb); 
	mac_enc(label); /* encreasing */

	if(!mac_valid(label)) {
		errno = ENOENT;
		mac_free(label);
		return (NULL);
	} else 
		return (label);
}

int	mac_set_fd(int fildes, mac_t label) // kernel
{
	int error;

	if(!mac_valid(label)) {
		errno = EINVAL;
		return (-1);
	}

	error = mac_set_fd_sc(fildes, label);

	return (error);
}

int	mac_set_file(const char *path_p, mac_t label) // kernel
{
	int error;

	if(!mac_valid(label)) {
		errno = EINVAL;
		return (-1);
	}

	error = mac_set_file_sc(path_p, label);

	return (error);
}

/* Setting proc mac label. If invoked by root, then setting max mac label
 * else - setting current mac label.
 */

int	mac_set_proc(mac_t label) // kernel
{
	int error;

	if(!mac_valid(label)) {
		errno = EINVAL;
		return (-1);
	}

	error = mac_set_proc_sc(label);

	return (error);
}

ssize_t	mac_size(mac_t label) // ??? libposix1e
{
	return sizeof(mac);
}

char * mac_to_text(mac_t label, size_t *len_p) // libposix1e
{
	char *buf;

	if(!mac_valid(label)) {
		errno = EINVAL;
		return (NULL);
	}
	if(len_p != NULL)
		buf = malloc(*len_p+1);
	else buf = malloc( 4 ); /* typical length */

	if(buf == NULL) {
		errno = ENOMEM;
		return (NULL);
	}

	sprintf(buf,"%d\0", (int)*label);

	if(!errno) return (buf);
	else return (NULL);
}

int	mac_valid(mac_t label) // libposix1e
{
	/*      = MAC_SYS_EMPTY */
	/* The only invalid label value is MAC_SYS_EMPTY */
	return (*label != (mac)MAC_SYS_EMPTY);
}

