/*
 *
 * POSIX 1e Capabilities
 *
 */

#ifndef CAPABILITY_H
#define CAPABILITY_H

#include <sys/types.h>
#include <sys/param.h>
#include <sys/capdef.h>

#define __CAP_MASK_LEN 2 /* 1 - POSIX 1E capabilities, 
			  * 2 - system capabilities 
			  */

typedef u_int32_t	cap_value_t;
typedef u_char	cap_flag_t;
typedef u_char	cap_flag_value_t;

struct capability {
	cap_value_t	effective[__CAP_MASK_LEN];
	cap_value_t	permitted[__CAP_MASK_LEN];
	cap_value_t	inheritable[__CAP_MASK_LEN];
};

/* set a bit in capability-><something> 
 *              ^^^^^^^^^^^^^^^^^^^^^^^ - this is (mask)
 * (cap) is defined in capdef.h
 * SYSTEM_CAPABILITY show wheather we have to set a bit in system capabilities
 * or in posix ones
 */

#define SET_CAPABILITY(mask,cap) { \
	mask[SYSTEM_CAPABILITY & cap] |= cap; \
	}

/* Check if named capability is set */

#define IS_CAP_SET(p,cap) \
	(p->pc_p1ea->caps.effective[SYSTEM_CAPABILITY & cap] & cap)

/* Check if capability sets(bitmasks) are equal */

#define IS_CAP_EQUAL(cap_a,cap_b) \
	((((cap_a[0] & cap_b[0]) == cap_a[0]) && \
	 ((cap_a[0] & cap_b[0]) == cap_b[0])) && \
	(((cap_a[1] & cap_b[1]) == cap_a[1]) && \
	 ((cap_a[1] & cap_b[1]) == cap_b[1])))

/* Check if it is allowed to set capabilities */
/* scap - set caps, ecap - existing cap */

#define IS_CAP_SET_ALLOWED(scap,ecap) \
	(((ecap[0]->permited & scap[0]->effective) == scap[0]->effective) && \
	((ecap[1]->permited & scap[1]->effective) == scap[1]->effective))

typedef struct capability cap;
typedef cap * cap_t;

extern cap _cap_max;

#ifdef _POSIX_CAP

#ifndef KERNEL

int	cap_clear(cap_t cap_p ); 
ssize_t cap_copy_ext( void *ext_p, cap_t cap_p, ssize_t size );
cap_t	cap_copy_int( const void *ext_p );
cap_t	cap_dup( cap_t cap_p );
int 	cap_free( void *obj );
cap_t	cap_from_text( const char *buf_p );
cap_t	cap_get_fd( int fd );
cap_t	cap_get_file( const char *path_p );
int	cap_get_flag( cap_t cap_p, cap_value_t cap, cap_flag_t flag, 
			cap_flag_value_t *value_p );
cap_t	cap_get_proc( void );
cap_t	cap_init( void );
int	cap_set_fd( int fd, cap_t cap_p );
int	cap_set_file( const char *path_p, cap_t cap_p );
int	cap_set_flag( cap_t cap_p, cap_flag_t flag, int ncap, 
			cap_value_t caps[], cap_flag_value_t value );
int	cap_set_proc( cap_t cap_p );
ssize_t	cap_size( cap_t cap_p );
char *	cap_to_text( cap_t cap_p, size_t *len_p );

#else /* KERNEL */

int get_cap(struct proc *, struct vnode *, cap_t);
int set_cap(struct proc *, struct vnode *, cap_t);

#endif /* KERNEL */

#else /* _POSIX_CAP */
#endif /* _POSIX_CAP */

#endif /* CAPABILITY_H */
