
/* 
 * This is derived from posix.1e draft and <linux/capability.h>
 */

#ifndef CAPDEF_H
#define CAPDEF_H

/* Table 25-3 contains cap_flag_t values for the cap_get_flag() and
 * cap_set_flag() functions.
 */
#define CAP_EFFECTIVE		1 /* Specifies the effective flag. */
#define CAP_INHERITABLE		2 /* Specifies the inheritable flag. */
#define CAP_PERMITTED		3 /* Specifies the permitted flag. */

/* Table 25-4 contains cap_flag_value_t values for the cap_get_flag() and
 * cap_set_flag () functions.
 */

#define CAP_CLEAR 0	/* The flag is cleared/disabled. */
#define CAP_SET 1	/* The flag is set/enabled. */


/* Table 25-5 through Table 25-8 contains cap_value_t values for cap_get_flag()
 * and cap_set_flag (). Note that the description of each capability specifies
 * exactly what restriction the capability is intended to affect. Possession of
 * a capability that overrides one restriction should not imply that any other
 * restrictions are overridden. For example, possession of the CAP_DAC_OVERRIDE
 * capability should not imply that a process can read files with MAC labels
 * that dominate that of the process, nor should it override any restrictions
 * that the file owner ID match the user ID of the process.
 */

/* If the {_POSIX_CAP} system configuration option is defined, the 
 * implementation shall define at least the following set of cap_value_t 
 * values:
 */

/* In a system in which the {_POSIX_CHOWN_RESTRICTED} option is defined, 
 * this capability shall override the restriction that a process cannot 
 * change  the user ID of a file it owns and the restriction that the group ID 
 * supplied to the chown() function shall be equal to either the group ID or 
 * one of the supplementary group IDs of the calling process.
 */

#define CAP_CHOWN		0x80000000

/* This capability shall override file mode execute access restrictions when 
 * accessing an object, and, if the {_POSIX_ACL} option is defined, this 
 * capability shall override the ACL execute access restrictions when 
 * accessing an object.
 */

#define CAP_DAC_EXECUTE 	0x40000000

/* This capability shall override file mode write access restrictions when 
 * accessing an object, and, if the {_POSIX_ACL} option is defined, this 
 * capability shall override the ACL write access restrictions when accessing 
 * an object.
 */

#define CAP_DAC_WRITE		0x20000000

/* This capability shall override file mode read and search access restrictions
 * when accessing an object, and, if the {_POSIX_ACL} option is defined, this
 * capability shall override the ACL read and search access restrictions when
 * accessing an object.
 */

#define CAP_DAC_READ_SEARCH	0x10000000

/* This capability overrides the requirement that the user ID associated with 
 * a process be equal to the file owner ID, except in the cases where the
 * CAP_FSETID capability is applicable. In general, this capability, when
 * effective, will permit a process to perform all the functions that any
 * file owner would have for their files.
 */

#define CAP_FOWNER		0x08000000

/* This capability shall override the following restrictions: that the
 * effective user ID of the calling process shall match the file owner when
 * setting the set-user-ID (S_ISUID) and set-group-ID (S_ISGID) bits on that
 * file; that the effective group ID or one of the supplementary group IDs
 * of the calling process shall match the group ID of the file when setting
 * the set-group-ID bit of that file; and that the set-user-ID and set-group-ID
 * bits of the file mode shall be cleared upon successful return from chown().
 */

#define CAP_FSETID		0x04000000

/* This capability shall override the restriction that the real or effective
 * user ID of a process sending a signal must match the real or effective
 * user ID of the receiving process.
 */

#define CAP_KILL		0x02000000

/* This capability overrides the restriction that a process can not create
 * or delete a hard link to a directory.
 */

#define CAP_LINK_DIR		0x01000000

/* This capability shall override the restriction that a process cannot set
 * the file capability state of a file.
 */

#define CAP_SETFCAP		0x00800000

/* This capability shall override the restriction in the setgid() function
 * that a process cannot change its real group ID or change its effective
 * group ID to a value other than its real group ID. If {_POSIX_SAVED_IDS}
 * is defined, then this capability also overrides any restrictions on setting
 * the saved set group-ID to a value other than the current real or saved
 * set-group ID.
 */

#define CAP_SETGID		0x00400000

/* This capability shall override the restriction in the setuid() function
 * that a process cannot change its real user ID or change its effective user
 * ID to a value other than the current real user ID. If {_POSIX_SAVED_IDS}
 * is defined, then this capability also overrides any restrictions on setting
 * the saved set-user ID.
 */

#define CAP_SETUID		0x00200000

/* If the {_POSIX_MAC} system configuration option is defined, the
 * implementation shall define at least the following set of cap_value_t 
 * values:
 */

/* This capability shall override the restriction that no process may
 * downgrade the MAC label of a file.
 */

#define CAP_MAC_DOWNGRADE	0x00100000

/* This capability shall override mandatory read access restrictions when
 * accessing objects.
 */

#define CAP_MAC_READ		0x00080000

/* This capability shall override the restriction that a process may not
 * modify its own MAC label.
 */

#define CAP_MAC_RELABEL_SUBJ	0x00040000

/* This capability shall override the restriction that no process may upgrade
 * the MAC label of a file.
 */

#define CAP_MAC_UPGRADE		0x00020000

/* This capability shall override mandatory write access restrictions when
 * accessing objects.
 */

#define CAP_MAC_WRITE		0x00010000


/* If the {_POSIX_INF} system configuration option is defined, the
 * implementation shall define at least the following set of cap_value_t 
 * values:
 */

/* This capability shall override the requirement that an object's 
 * information label shall automatically float when a write operation 
 * is performed by a process.
 */

#define CAP_INF_NOFLOAT_OBJ	0x00008000

/* This capability shall override the requirement that a process information 
 * label shall automatically float when a read or execute operation is 
 * performed on an object.
 */

#define CAP_INF_NOFLOAT_SUBJ	0x00004000

/* This capability shall override the restriction against changing the 
 * information label of an object.
 */ 

#define CAP_INF_RELABEL_OBJ	0x00002000

/* This capability shall override the restriction that a process may not 
 * modify its own information label in violation of the information labeling 
 * policy.
 */

#define CAP_INF_RELABEL_SUBJ	0x00001000


/* If the {_POSIX_AUD} system configuration option is defined, the
 * implementation shall define at least the following set of cap_value_t 
 * values:
 */

/* This capability shall override the restriction that a process cannot 
 * modify audit control parameters.
 */

#define CAP_AUDIT_CONTROL	0x00000800

/* This capability shall override the restriction that a process cannot 
 * write data into the system audit trail.
 */

#define CAP_AUDIT_WRITE		0x00000400



/**
 **
 ** Linux-specific capabilities
 **
 ** Let's spread it to FreeBSD ;-)
 **
 **/

#define SYSTEM_CAPABILITY	0x00000001

/* Transfer any capability in your permitted set to any pid,
 * remove any capability in your permitted set from any pid 
 */

#define CAP_SETPCAP          0x80000001

/* Allow modification of S_IMMUTABLE and S_APPEND file attributes */

#define CAP_LINUX_IMMUTABLE  0x40000001

/* Allows binding to TCP/UDP sockets below 1024 
 * Allows binding to ATM VCIs below 32
 */

#define CAP_NET_BIND_SERVICE 0x20000001

/* Allow broadcasting, listen to multicast */

#define CAP_NET_BROADCAST    0x10000001

/* Allow interface configuration 
 * Allow administration of IP firewall, masquerading and accounting 
 * Allow setting debug option on sockets 
 * Allow modification of routing tables 
 * Allow setting arbitrary process / process group ownership on sockets 
 * Allow binding to any address for transparent proxying 
 * Allow setting TOS (type of service) 
 * Allow setting promiscuous mode 
 * Allow clearing driver statistics 
 * Allow multicasting 
 * Allow read/write of device-specific registers 
 * Allow activation of ATM control sockets
 */

#define CAP_NET_ADMIN        0x08000001

/* Allow use of RAW sockets 
 * Allow use of PACKET sockets
 */

#define CAP_NET_RAW          0x04000001

/* Allow locking of shared memory segments 
 * Allow mlock and mlockall (which doesn't really have anything to do
 * with IPC)
 */

#define CAP_IPC_LOCK         0x02000001

/* Override IPC ownership checks */

#define CAP_IPC_OWNER        0x01000001

/* Insert and remove kernel modules */

#define CAP_SYS_MODULE       0x00800001

/* Allow ioperm/iopl access 
 * Allow sending USB messages to any device via /proc/bus/usb
 */

#define CAP_SYS_RAWIO        0x00400001

/* Allow use of chroot() */

#define CAP_SYS_CHROOT       0x00200001

/* Allow ptrace() of any process */

#define CAP_SYS_PTRACE       0x00100001

/* Allow configuration of process accounting */

#define CAP_SYS_PACCT        0x00080001

/* Allow configuration of the secure attention key 
 * Allow administration of the random device 
 * Allow device administration (mknod)
 * Allow examination and configuration of disk quotas 
 * Allow configuring the kernel's syslog (printk behaviour) 
 * Allow setting the domainname 
 * Allow setting the hostname 
 * Allow calling bdflush() 
 * Allow mount() and umount(), setting up new smb connection 
 * Allow some autofs root ioctls 
 * Allow nfsservctl 
 * Allow VM86_REQUEST_IRQ 
 * Allow to read/write pci config on alpha 
 * Allow irix_prctl on mips (setstacksize) 
 * Allow flushing all cache on m68k (sys_cacheflush) 
 * Allow removing semaphores 
 * Used instead of CAP_CHOWN to "chown" IPC message queues, semaphores
 * and shared memory 
 * Allow locking/unlocking of shared memory segment 
 * Allow turning swap on/off 
 * Allow forged pids on socket credentials passing 
 * Allow setting readahead and flushing buffers on block devices 
 * Allow setting geometry in floppy driver 
 * Allow turning DMA on/off in xd driver 
 * Allow administration of md devices (mostly the above, but some extra ioctls) 
 * Allow tuning the ide driver 
 * Allow access to the nvram device 
 * Allow administration of apm_bios, serial and bttv (TV) device 
 * Allow manufacturer commands in isdn CAPI support driver 
 * Allow reading non-standardized portions of pci configuration space 
 * Allow DDI debug ioctl on sbpcd driver 
 * Allow setting up serial ports 
 * Allow sending raw qic-117 commands 
 * Allow enabling/disabling tagged queuing on SCSI controllers and sending
 * arbitrary SCSI commands 
 * Allow setting encryption key on loopback filesystem
 */

#define CAP_SYS_ADMIN        0x00040001

/* Allow use of reboot() */

#define CAP_SYS_BOOT         0x00020001

/* Allow raising priority and setting priority on other (different
 * UID) processes 
 * Allow use of FIFO and round-robin (realtime) scheduling on own
 * processes and setting the scheduling algorithm used by another
 * process.
 */

#define CAP_SYS_NICE         0x00010001

/* Override resource limits. Set resource limits. 
 * Override quota limits. 
 * Override reserved space on ext2 filesystem 
 * NOTE: ext2 honors fsuid when checking for resource overrides, so 
 * you can override using fsuid too 
 * Override size restrictions on IPC message queues 
 * Allow more than 64hz interrupts from the real-time clock 
 * Override max number of consoles on console allocation 
 * Override max number of keymaps
 */

#define CAP_SYS_RESOURCE     0x00008001

/* Allow manipulation of system clock 
 * Allow irix_stime on mips 
 * Allow setting the real-time clock
 */

#define CAP_SYS_TIME         0x00004001

/* Allow configuration of tty devices 
 * Allow vhangup() of tty
 */

#define CAP_SYS_TTY_CONFIG   0x00002001

#endif /* CAPDEF_H */
