/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: main.c,v 1.4 1998/07/13 22:35:15 robert Exp $
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "types.h"
#include "const.h"
#include "globals.h"
#include "checker.h"
#include "interface.h"
#include "utils.h"

extern int policyparse(void);
extern int implementationparse(void);

void printargs(void) {
	fprintf(stderr, "suidcontrol [-c conffilename] [-i implementationfilename] [-parse_debug]\n");
	fprintf(stderr, "            [-notdefaultdefault] [-xalsoifdefault]\n");
}

#define ARGWARN(line...) fprintf(stderr, ## line)
#define MAX_STRING_LEN 120

void main(int argc, char *argv[]) { 

	struct setstr *set;

	char *conffilename = "policy";		/* default config file name */
	char *implementationfilename = "implementation";
		/* default script file name */
	char s[MAX_STRING_LEN];
	int i, done;


	for (i=1; i<argc; i++) {
		if (!strcmp("-c", argv[i])) {
			i++;
			if (i==argc) {
				ARGWARN("Argument error: '-c' without conffilename\n");
				printargs();
				exit(-1);
			}
			conffilename = argv[i];
		} else
		if (!strcmp("-i", argv[i])) {
			i++;
			if (i == argc) {
				ARGWARN("Argument error: '-i' without outfilename\n");
				printargs();
				exit(-1);
			}
			implementationfilename = argv[i];
		} else 
		if (!strcmp("-parse_debug", argv[i])) {
			parse_debug = 1;
		} else
		if (!strcmp("-notdefaultdefault", argv[i])) {
			notdefaultdefault = 1;
		} else
		if (!strcmp("-xalsoifdefault", argv[i])) {
			xalsoifdefault = 1;
		} else
		if(!strcmp("-help", argv[i])) {
			printargs();
			exit(0);
		} else {
			ARGWARN("Argument error: '%s' unrecognized argument\n", argv[i]);
			printargs();
			exit(-1);
		}
	}
	
	policyin = fopen(conffilename, "r");
	if (!policyin) {
		perror("Error opening configuration file");
		exit(-1);
	}
	source = conffilename;

	i = policyparse();
	if (i) {
		fprintf(stderr, "Configuration file parse failed.\n");
		exit(-1);
	}

	fclose(policyin);

	fprintf(stderr, "Configuration file successfully parsed.\n");

	for (set = topset; (set); set = set->next) {
		if ((!organize(set)) || (!checkset(set))) {
			fprintf(stderr, "Configuration file interpretation failed.\n");
			exit(-1);
		}
		printf("Set: '%s'\n", set->name);
	}

	implementationin = fopen(implementationfilename, "r");
	if (!implementationin) {
		perror("Error opening configuration file");
		if (errno != ENOENT) {
			exit(-1);
		}
		/* file not found is ok -- we will create when done */
		fprintf(stderr, "Implementation file not found\n");
		topimpl = 0;
	}
	else
	{
		source = implementationfilename;

		i = implementationparse();
		if (i) {
			fprintf(stderr, "Implementation file parse failed.\n");
			exit(-1);
		}

		fclose(implementationin);

		fprintf(stderr, "Implementation file successfully parsed.\n");
	}

	interface_init();

	interface_notify("suidcontrol v0.1 -- Copyright (c) 1998 Robert "
			"Watson, all rights reserved.\n"
			"Please read the license.suidcontrol "
			"file to find out more.");

	if (topimpl) {
		/* implementation file was found and parsed */
		if (!interface_yesno("Use existing implementation values?",
			"Existing implementation file was found. "
			" Would you like to use its current contents for "
			" initial values? (yes to use current values, no to "
			" reset to system defaults) ")) {
			topimpl = 0; /* XXXX big memory leak */
		} else {
			fprintf(stderr, "using existing impl\n");
		}
	} else {
		/* implementation file was not found */
		if (!interface_yesno("Use system defaults for initial values?",
			"The implementation file you specified "
			"was not found.  Use system defaults for initial "
			"values?  (yes uses system defaults, no to exit)")) {
			interface_destroy();
			fprintf(stderr, "Done.\n");
			exit(-1);
		}
	}

	/* incorporate any defaults desired */
	if (!incorporate_impl(topimpl)) {
		fprintf(stderr, "Unable to apply defaults to configuration.\n");		exit(-1);
	}

	done = 0;
	while (!done) {
		i = interface_selectset();
		if (i == 0) {
			/* success */
			if (interface_yesno("Save and Exit?",
				"Save work and exit? (Yes exits and saves; "
				"No returns to the menu without saving)")) {
				/* save */
				i = save_impl(implementationfilename);
				if (i) {
					snprintf(s, MAX_STRING_LEN,
						"Unable to save: %s",
						strerror(i));
					interface_notify(s);
				} else {
					done = 1;
				}
			} else {
				/* continue w/no action */
			}
		} else
		if (i == 1) {
			/* cancel */
			if (interface_yesno("Cancel and lose changes?",
				"Exit without saving changes?  (Yes exits "
				"without saving changes; no returns to the "
				"menu.)")) {
				/* cancel and exit */
				done = 1;
			} else {
				/* continue */
			}
		} else
		if (i == -1) {
			/* error */
			interface_notify("Ooops!  "
					"Something went wrong in libdialog.  "
					"I'm going to exit and lose any "
					"changes, but won't change any files."
					"  Sorry about that!");
			done = 1;
		}
	}

	interface_destroy();
}

