/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: tf_util_token.c,v 1.1 1998/07/06 00:18:41 robert Exp $
 */

#include <sys/types.h>
#include <sys/time.h>
#include <machine/types.h>
#include <unistd.h>

#include <stdio.h>
#include <string.h>

#include "/usr/tmp/cvs/usr/robert/fbsd-tokens/impl/common/ktoken_const.h"
#include "/usr/tmp/cvs/usr/robert/fbsd-tokens/impl/common/ktoken_types.h"
#include "/usr/tmp/cvs/usr/robert/fbsd-tokens/impl/common/ktoken_structs.h"
#include "/usr/tmp/cvs/usr/robert/fbsd-tokens/impl/userland/libktoken.h"

#include "/usr/src/crypto/kerberosIV/lib/krb/krb_locl.h"

#define DEBUG(line...) /* printf(## line) */

static tokenid_t primary_token;
static tokenid_t current_token;
static int tf_init_called=0;

int tf_init(char *tf_name, int rw) {
    /* verify in a PAG */
    pagid_t pagid;
    struct utokenstr u;
    u_int mask;
    pagid = t_getpag();
    if (pagid == BADPAGID) {
	DEBUG("tf_init: no pag\n");
	return(TKT_FIL_ACC);
    }

    DEBUG("tf_init: found pag %lu\n", pagid);

    /* retrieve a tgt from pag */
    
    u.major = TOKEN_MAJ_KERBEROS4;
    u.minor = TOKEN_KRB4_PRIMARY;
    u.minorminor = 0;

    mask = TOKEN_FIELD_MAJOR | TOKEN_FIELD_MINOR | TOKEN_FIELD_MINORMINOR;

    primary_token = t_findtoken(0, mask, &u);
    current_token = 0;

    tf_init_called = 1;
    if (primary_token == BADTOKENID) {
	/* no matching token */
	DEBUG("tf_init: no primary krbIV token found\n");
	return(NO_TKT_FIL);
    }

    /* otherwise, we have one */
    /* lock doesn't make sense or anything, just return */
    DEBUG("tf_init: primary krbIV token is %lu\n", primary_token);

    return(KSUCCESS);
}

int tf_create(char *tf_name) {
    /* if a primary_token already exists, delete it */
    /* then create a new primary token */
    /* then do a find -- if one already exists, we were raced, so delete
       ours and return a failure.  otherwise succeed */
    /* this comes of trying to a) have a header like the tf_file header
       and b) offer exclusive file-like semantics for it */

    tokenid_t tokenid;
    struct utokenstr u;
    u_int mask;
    int i;

/*    if (!tf_init_called) {
	return(TKT_FIL_INI);
    } */

    memset(&u, '\0', sizeof(u));
    strncpy(u.name, "", T_STRINGLEN_MAX); 
    strncpy(u.realm, "", T_STRINGLEN_MAX);
    u.rights = (T_R_READ |
		T_R_MODIFY |
		T_R_DELETE |
		T_R_INHERIT |
		T_R_TRANSFERABLE);
    u.major = TOKEN_MAJ_KERBEROS4;
    u.minor = TOKEN_KRB4_PRIMARY;
    u.minorminor = 0;
    u.publiclen = 0;
    u.privatelen = 0;
    u.createtime = time(0);
    u.expiretime = 0;
    
    mask = TOKEN_FIELD_MAJOR;

    /* delete old tokens first */

    while ((tokenid = t_findtoken(0, mask, &u)) != BADTOKENID) {
	i = t_deletetoken(tokenid);
	if (i) {
	    perror("tf_create.t_deletetoken");
	}
    }

    /* create new one */

    primary_token = t_createtoken(&u);
    if (primary_token == BADTOKENID) {
	perror("tf_create.t_createtoken");
	return(TKT_FIL_ACC);
    }

    /* is it the right one? */
    mask = (TOKEN_FIELD_MAJOR |
	    TOKEN_FIELD_MINOR); 
    tokenid = t_findtoken(0, mask, &u);
    if (tokenid != primary_token) {
	/* got back someone else's token */
	i = t_deletetoken(primary_token);
	if (i) {
	    perror("tf_create.t_deletetoken");
	}
	return(TKT_FIL_LCK);
    }
    /* all is well */
	tf_init_called = 1; /* XXXX */
    return(KSUCCESS);
}

int tf_get_pname(char *p) {
    struct utokenstr u;
    int i;

    if (!tf_init_called) {
	return(TKT_FIL_INI);
    }

    i = t_readtoken(primary_token, &u);
    if (i) {
	perror("tf_get_pname.t_readtoken");
	return(TKT_FIL_FMT);
    }

    if (strlen(u.name)+1 > ANAME_SZ) {
	return(TKT_FIL_FMT);
    }
    strcpy(p, u.name);
    return KSUCCESS;
}

int tf_put_pname(char *p) {
    int i;
    u_int mask;
    struct utokenstr u;

    if (!tf_init_called) {
	return(TKT_FIL_INI);
    }
    
    strncpy(u.name, p, T_STRINGLEN_MAX);
    mask = TOKEN_FIELD_NAME;

    i = t_modifytoken(primary_token, mask, &u);
    if (i) {
	perror("tf_put_pname.t_modifytoken");
	return(KFAILURE);
    }
    return KSUCCESS;
}

int tf_get_pinst(char *p) {
    struct utokenstr u;
    int i;

    if (!tf_init_called) {
	return(TKT_FIL_INI);
    }

    i = t_readtoken(primary_token, &u);
    if (i) {
	perror("tf_get_pinst.t_readtoken");
	return(TKT_FIL_FMT);
    }

    if (strlen(u.name)+1 > ANAME_SZ) {
	return(TKT_FIL_FMT);
    }
    strcpy(p, u.instance);
    return KSUCCESS;
}

int tf_put_pinst(char *p) {
    int i;
    u_int mask;
    struct utokenstr u;

    if (!tf_init_called) {
	return(TKT_FIL_INI);
    }
    
    strncpy(u.instance, p, T_STRINGLEN_MAX);
    mask = TOKEN_FIELD_NAME;

    i = t_modifytoken(primary_token, mask, &u);
    if (i) {
	perror("tf_put_pinst.t_modifytoken");
	return(KFAILURE);
    }
    return KSUCCESS;
}

int tf_get_cred(CREDENTIALS *c) {
    KTEXT ticket = &c->ticket_st; /* ptr to ticket */
    int i;
    u_int mask;
    struct utokenstr u; 

    if (!tf_init_called) {
	return(TKT_FIL_INI);
    }

again:
  
    mask = (TOKEN_FIELD_MAJOR |
	    TOKEN_FIELD_MINOR);

    u.major = TOKEN_MAJ_KERBEROS4;
    u.minor = TOKEN_KRB4_CRED;
    u.minorminor = 0;

    current_token = t_findtoken(current_token+1, mask, &u);
    if (current_token == BADTOKENID) {
	return(EOF);
    }

    i = t_readtoken(current_token, &u);
    if (i) {
	/* race on a token */
	perror("tf_get_cred.t_readtoken");
	goto again;
    }

    strncpy(c->service, u.name, SNAME_SZ);
    strncpy(c->instance, u.instance, INST_SZ);
    strncpy(c->realm, u.realm, REALM_SZ);

    if (u.privatelen != DES_KEY_SZ) {
	return(TKT_FIL_FMT);
    }

    memcpy(&(c->session), &(u.privatedat), DES_KEY_SZ);
    c->kvno = u.kvno; /* minorminor */
    c->lifetime = u.expiretime - u.createtime;
    c->issue_date = u.createtime;

    if (u.publiclen > MAX_KTXT_LEN) {
	return(TKT_FIL_FMT);
    }
    ticket->length = u.publiclen;
    memcpy(ticket->dat, &(u.publicdat), u.publiclen);
    
    return(KSUCCESS);
}

void tf_close(void) {
    /* noop */
    current_token = 0;
}

int tf_save_cred(char *service,
		 char *instance,
		 char *realm,
		 unsigned char *session,
		 int lifetime,
		 int kvno,
		 KTEXT ticket,
		 u_int32_t issue_date)
{
    struct utokenstr u;
    tokenid_t tokenid;

    memset(&u, '\0', sizeof(u));

    strncpy(u.name, service, T_STRINGLEN_MAX);
    strncpy(u.instance, instance, T_STRINGLEN_MAX);
    strncpy(u.realm, realm, T_STRINGLEN_MAX);
    memcpy(u.privatedat, session, DES_KEY_SZ);
    u.privatelen = DES_KEY_SZ;
    u.kvno = kvno; /* minorminor */
    u.expiretime = issue_date + lifetime;
    u.createtime = issue_date;
    if (ticket->length > T_DATALEN_MAX) {
	DEBUG("tf_save_cred: ticket too big\n");
	return(KFAILURE);
    }

    u.publiclen = ticket->length;
    memcpy(u.publicdat, ticket, ticket->length);

    u.rights = (T_R_READ |
		T_R_MODIFY | 
		T_R_DELETE |
		T_R_INHERIT |
		T_R_TRANSFERABLE);
    u.major = TOKEN_MAJ_KERBEROS4;
    u.minor = TOKEN_KRB4_CRED;

    tokenid = t_createtoken(&u);
    if (tokenid == BADTOKENID) {
	perror("tf_save_cred.t_createtoken");
	return(KFAILURE);
    }

    current_token = tokenid;
    return(KSUCCESS);
}

int
tf_setup(CREDENTIALS *cred, char *pname, char *pinst)
{
    int ret;
    ret = tf_create(tkt_string());
    if (ret != KSUCCESS)
	return ret;

    if (tf_put_pname(pname) != KSUCCESS ||
	tf_put_pinst(pinst) != KSUCCESS) {
	tf_close();
	return INTK_ERR;
    }

    ret = tf_save_cred(cred->service, cred->instance, cred->realm, 
		       cred->session, cred->lifetime, cred->kvno,
		       &cred->ticket_st, cred->issue_date);
    tf_close();
    return ret;
}

int
in_tkt(char *pname, char *pinst)
{
  int ret;
  
  ret = tf_create (tkt_string());
  if (ret != KSUCCESS)
    return ret;

    if (tf_put_pname(pname) != KSUCCESS ||
	tf_put_pinst(pinst) != KSUCCESS) {
	tf_close();
	return INTK_ERR;
    }

    tf_close();
    return KSUCCESS;
}
    

	
