/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: libktoken.c,v 1.6 1998/07/06 00:18:42 robert Exp $
 */

/* 
 * library support for kernel tokens
 */

#include <sys/types.h>
#include <sys/time.h>
#include <machine/types.h>
#include <sys/syscall.h>
#include <unistd.h>
#include <stdio.h>

#include "../common/ktoken_const.h"
#include "../common/ktoken_types.h"
#include "../common/ktoken_structs.h"
#include "../common/ktoken_syscall.h"
#include "libktoken.h"

/* this should not be here */
int t_syscallnum=210;

/* overide the default syscall number */
void t_setsyscallnum(int i) {
	t_syscallnum = i;
}

/* make this process join a new PAG; returns its pagid 
 * returns 0 on failure; errno may be checked for error condition
 */
pagid_t t_newpag() {
	int err = 0;
	struct t_syscall_args	sa;
	struct t_newpag_args	np;
	pagid_t	pagid;

	sa.whichfunction = TOKEN_SYSCALL_NEWPAG;
	sa.args = &np;
	np.p_pagid = &pagid;

	err = syscall(t_syscallnum, sa);

	if (err) return (0);
	else return(pagid);
}

/* return the current PAGs pagid; returns 0 on failure, errno
 * maye be checked for error condition
 */
pagid_t t_getpag() {
	int err = 0;
	struct t_syscall_args	sa;
	struct t_getpag_args	gp;
	pagid_t	pagid;

	sa.whichfunction = TOKEN_SYSCALL_GETPAG;
	sa.args = &gp;
	gp.p_pagid = &pagid;

	err = syscall(t_syscallnum, sa);

	if (err) return(0);
	else return(pagid);
}

/* read token 'tokenid' into a pre-allocated utokenstr buffer p_token.
 * returns !0 on failure, 0 on success.  Errno contains error if non-zero
 */
int t_readtoken(tokenid_t tokenid, struct utokenstr *p_token) {
	int err = 0;
	struct t_syscall_args	sa;
	struct t_readtoken_args	rt;
	
	sa.whichfunction = TOKEN_SYSCALL_READTOKEN;
	sa.args = &rt;
	rt.p_token = p_token;
	rt.tokenid = tokenid;

	err = syscall(t_syscallnum, sa);
	return(err);
}

/* modify token 'tokenid'.  Copy fields indicate in whichfields bitmask
 * from p_token, a pointer to a filled utokenstr.  Returns 0 on succesws,
 * non-zero on failure.  Check errno for error condition.
 */
int t_modifytoken(tokenid_t tokenid,
			int whichfields,
			struct utokenstr *p_token) {
	int err = 0;
	struct t_syscall_args	sa;
	struct t_modifytoken_args	mt;

	sa.whichfunction = TOKEN_SYSCALL_MODIFYTOKEN;
	sa.args = &mt;
	mt.tokenid = tokenid;
	mt.whichfields = whichfields;
	mt.p_token = p_token;

	err = syscall(t_syscallnum, sa);
	return(err);
}

/* Create a new token based on the provided model token -- 0 on success,
 * non-zero on failure.  Check errno for details.
 */
tokenid_t t_createtoken(struct utokenstr *p_token) {
	int	err = 0;
	struct t_syscall_args	sa;
	struct t_createtoken_args	ct;
	tokenid_t	tokenid;

	sa.whichfunction = TOKEN_SYSCALL_CREATETOKEN;
	sa.args = &ct;
	ct.p_tokenid = &tokenid;
	ct.p_token = p_token;

	err = syscall(t_syscallnum, sa);
	if (err) return (0);
	else return(tokenid);
}

/* create a reflection of token 'tokenid'.  Return a new tokenid of
 * the reflection.  Returns 0 on failure; check errno for details
 */
tokenid_t t_reflecttoken(tokenid_t tokenid1) {
	int	err = 0;
	struct t_syscall_args	sa;
	struct t_reflecttoken_args	rt;
	tokenid_t	tokenid2;

	sa.whichfunction = TOKEN_SYSCALL_REFLECTTOKEN;
	sa.args = &rt;
	rt.tokenid1 = tokenid1;
	rt.p_tokenid2 = &tokenid2;

	err = syscall(t_syscallnum, sa);
	if (err) return 0;
	else return(tokenid2);
}

/* delete token 'tokenid' -- returns 0 on success, non-0 on failure.
 * check errno for details.
 */
int t_deletetoken(tokenid_t tokenid) {
	int	err = 0;
	struct t_syscall_args	sa;
	struct t_deletetoken_args	dt;

	sa.whichfunction = TOKEN_SYSCALL_DELETETOKEN;
	sa.args = &dt;
	dt.tokenid = tokenid;

	err = syscall(t_syscallnum, sa);
	return err;
}

/* find a token matching whichfields fields of p_token, with minimum
 * tokenid of mintokenid.  Returns a tokenid on success, 0 on failure.
 * check errno for details.
 */
tokenid_t t_findtoken(tokenid_t mintokenid,
			u_int32_t whichfields,
			struct utokenstr *p_token) {
	int	err = 0;
	struct t_syscall_args	sa;
	struct t_findtoken_args	ft;
	tokenid_t	tokenid;

	sa.whichfunction = TOKEN_SYSCALL_FINDTOKEN;
	sa.args = &ft;
	ft.mintokenid = mintokenid;
	ft.whichfields = whichfields;
	ft.p_tokenid = &tokenid;
	ft.p_token = p_token;

	err = syscall(t_syscallnum, sa);
	if (err) return(0);
	else return (tokenid);
}

/* set the uid to one from the provided token, if permitted
 */
int t_setuidtoken(tokenid_t tokenid)
{
	int	err = 0;
	struct t_syscall_args	sa;
	struct t_setuidtoken_args	st;

	sa.whichfunction = TOKEN_SYSCALL_SETUIDTOKEN;
	sa.args = &st;
	st.tokenid = tokenid;

	err = syscall(t_syscallnum, sa);
	return(err);
}	

/* set debugging level for lkm.  Only super-user may call. 0 on
 * success, non-zero on failure -- check errno for details 
 */
int t_setdebug(int debuglevel)
{
	int	err = 0;
	struct t_syscall_args   sa;
        struct t_setdebug_args	da;

	sa.whichfunction = TOKEN_SYSCALL_SETDEBUG;
	sa.args = &da;
	da.debug = debuglevel;

	err = syscall(t_syscallnum, sa);
	return err;
}

/* special debugging call to imbue tokend token -- will go away
 * eventually
 */
int t_settokend(void)
{
	int err=0;
	struct t_syscall_args	sa;
	struct t_settokend_args	st;

	sa.whichfunction = TOKEN_SYSCALL_SETTOKEND;
	sa.args = &st;
	st.dummy = 0;

	err = syscall(t_syscallnum, sa);
        return err;
}

char *TOKEN_CRT_NULL_str = "null";
char *TOKEN_CRT_KERNEL_str = "kernel";
char *TOKEN_CRT_PROCESS_str = "process";
char *TOKEN_CRT_TOKEND_str = "tokend";
char *TOKEN_CRT_UNKNOWN_str = "unknown";

static char *t_tokenorigintoname(int origin) {
	switch(origin) {
		case TOKEN_CRT_NULL: return TOKEN_CRT_NULL_str;
		case TOKEN_CRT_KERNEL: return TOKEN_CRT_KERNEL_str;
		case TOKEN_CRT_PROCESS: return TOKEN_CRT_PROCESS_str;
		case TOKEN_CRT_TOKEND: return TOKEN_CRT_TOKEND_str;
		default: return TOKEN_CRT_UNKNOWN_str;
	}
}

char *TOKEN_MAJ_NULL_str = "null";
char *TOKEN_MAJ_LOCALPRIV_str = "localpriv";
char *TOKEN_MAJ_KERBEROS4_str = "kerberos4";
char *TOKEN_MAJ_KERBEROS5_str = "kerberos5";
char *TOKEN_MAJ_CODA_str = "coda";
char *TOKEN_MAJ_RSA_str = "rsa";
char *TOKEN_MAJ_IPSEC_str = "ipsec";
char *TOKEN_MAJ_SSHAGENT_str = "ssh";
char *TOKEN_MAJ_UNKNOWN_str = "unknown";

static char *t_tokenmajtoname(int type) {
	switch(type) {
		case TOKEN_MAJ_NULL: return TOKEN_MAJ_NULL_str;
		case TOKEN_MAJ_LOCALPRIV: return TOKEN_MAJ_LOCALPRIV_str;
		case TOKEN_MAJ_KERBEROS4: return TOKEN_MAJ_KERBEROS4_str;	
		case TOKEN_MAJ_KERBEROS5: return TOKEN_MAJ_KERBEROS5_str;
		case TOKEN_MAJ_CODA: return TOKEN_MAJ_CODA_str;
		case TOKEN_MAJ_RSA: return TOKEN_MAJ_RSA_str;
		case TOKEN_MAJ_IPSEC: return TOKEN_MAJ_IPSEC_str;
		case TOKEN_MAJ_SSHAGENT: return TOKEN_MAJ_SSHAGENT_str;
		default: return TOKEN_MAJ_UNKNOWN_str;
	}
}

/* visual representation of p_token printed 
 */
void t_tokenprettyprint(struct utokenstr *p_token) {
	int type;
	printf("TokenID: %lu       Name: %s.%s@%s\n", p_token->tokenid,
		p_token->name, p_token->instance, p_token->realm);

	printf(" Rights: ");
	if (p_token->rights & T_R_READ) printf("R") ;
	else printf("r");
	
	if (p_token->rights & T_R_MODIFY) printf("M") ;
	else printf("m");
	
	if (p_token->rights & T_R_DELETE) printf("D") ;
	else printf("d");
	
	if (p_token->rights & T_R_INHERIT) printf("I") ;
	else printf("i");
	
	if (p_token->rights & T_R_EXPIRE) printf("E") ;
	else printf("e");
	
	if (p_token->rights & T_R_TRANSFERABLE) printf("T") ;
	else printf("t");
	
	if (p_token->rights & T_R_TRANSFERABLEONCE) printf("1") ;
	else printf(" ");

	type = p_token->major;
	if (type < 0) {
		type = (-1) * type;
		printf("     Type: %s (%d,%d,%d) (reflection)\n",
			t_tokenmajtoname(type),
			p_token->major, p_token->minor, p_token->minorminor);
	} else {
		printf("     Type: %s (%d,%d,%d)\n",
			t_tokenmajtoname(type),
			p_token->major, p_token->minor, p_token->minorminor);
	}
	printf(" Pub length: %d       Priv length: %d\n", p_token->publiclen,
		p_token->privatelen);
	printf(" Createtime: %lu       Expiretime: %lu\n", p_token->createtime,
		p_token->expiretime);
	printf(" Creator: %s (%d)\n", t_tokenorigintoname(p_token->creator),
		p_token->creator);
}

