#ifdef DEBUG
#include <iostream>
#endif

#include <sstream>

#include "manager.h"

using std::string;
using std::cout;
using std::endl;

namespace ClusterManagement {

void Manager::joinMess (const char* sender, const string& name, int numMembers,
                        const char groupMembers[][MAX_GROUP_NAME]) {

  if (!_groupMembers.size()) {
    _groupMembers.reserve(numMembers > 10 ? numMembers : 10);
    for (int i = 0; i < numMembers; i++)
      _groupMembers.push_back(groupMembers[i]);
  }
  else {
    if (_groupMembers.size() == _groupMembers.capacity())
      _groupMembers.reserve(_groupMembers.size() << 1);
    _groupMembers.push_back(name);
  }

  //         :)
  if (numMembers > 1)
    if (name != groupMembers[0]) {
      if (!::strcmp(_privateGroup, groupMembers[0]))
        receiveNodeInit(name);
    }
    else if (!::strcmp(_privateGroup, groupMembers[1]))
      receiveNodeInit(name);
}

void Manager::leaveMess (const char* sender, const string& name, int numMembers,
                         const char groupMembers[][MAX_GROUP_NAME]) {
#ifdef DEBUG
  std::cout << "Member out!" << std::endl;
#endif

  receiveNodeLeave(name);
}

void Manager::disconnectMess (const char* sender, const string& name, int numMembers,
                              const char groupMembers[][MAX_GROUP_NAME]) {
#ifdef DEBUG
  std::cout << "Member out by disconnect!" << std::endl;
#endif

  receiveNodeLeave(name);
}

void Manager::networkMess (const char* sender, const std::string& name, int numMembers,
                           const char groupMembers[][MAX_GROUP_NAME]) {

  for (int i = 0; i < numMembers; i++) {
    if (std::find(_groupMembers.begin(), _groupMembers.end(), groupMembers[i]) ==
        _groupMembers.end())
      joinMess (sender, groupMembers[i], numMembers, groupMembers);
  }

  for (size_t i = 0; i < _groupMembers.size(); i++) {
    int j = 0;
    for (; j < numMembers; j++)
      if (!::strcmp(_groupMembers[i].c_str(), groupMembers[j]))
        break;
    if (j == numMembers)
      receiveNodeLeave(_groupMembers[i]);
  }
}


void Manager::regularMess (const char* sender, int numGroups,
                           const char groupMembers[][MAX_GROUP_NAME], int16_t mType, int endian,
                           int maxMessLen, const char* message) {
  switch (mType) {
    case NEW_RESOURCE:
      receiveNewResource (string(message, 0, maxMessLen));
      break;
    case CHANGE_RESOURCE:
      receiveChangeResource (string(message, 0, maxMessLen));
      break;
    case DEL_RESOURCE:
      receiveDelResource (string(message, 0, maxMessLen));
      break;
    case CLUSTER_RESOURCES_LIST:
      if (::strlen(message))
        receiveResourceTable (string(message, 0, maxMessLen));
      break;
    case GET_RESOURCES_LIST:
      if (!::strcmp(_privateGroup, _groupMembers[0].c_str()))
        receiveNodeInit(sender);
      break;
    default:
#ifdef DEBUG

      std::cerr << "Unknown message type (regular message)" << std::endl;
#endif

      break;
  }
}

void Manager::addResource(const string& name, int priority) {
  if (_rt.addResource (_privateGroup, name, priority)) {
    std::ostringstream message;
    message << _privateGroup << std::endl << name << std::endl << priority;
    sendMessage (NEW_RESOURCE, message.str());
    recalculateState(name);
  }
}

void Manager::receiveNodeLeave(const string& node) {
#ifdef DEBUG
  std::cout << "Node leave: " << node << std::endl;
#endif

  _rt.delNode(node);
  _groupMembers.erase(remove
                      (_groupMembers.begin(), _groupMembers.end(), node));
  recalculateState();
}

void Manager::receiveNodeInit(const string& node) {
  std::ostringstream message;
#ifdef DEBUG

  std::cout << "Node inited: " << node << std::endl;
#endif

  _rt.sort();
  ResTable table = _rt.getResTable();
  for (Rcit i = table.begin(); i != table.end(); i++)
    message << i->getNode() << " " << i->getName() << " " <<
    i->getPriority() << "\n";
  sendMessage (CLUSTER_RESOURCES_LIST, message.str(), node);
}

void Manager::receiveResourceTable(const string& message) {
  for (size_t from = 0, index = message.find('\n', from);
       index != string::npos; from = index + 1,
       index = message.find('\n', from))
    receiveNewResource(string(message, from, index - from));
}

void Manager::receiveNewResource(const string& message) {
  std::istringstream m(message);
  string node;
  string name;
  int priority;

  m >> node >> name >> priority;
  if (std::find(_groupMembers.begin(), _groupMembers.end(), node) ==
      _groupMembers.end()) {
#ifdef DEBUG
    std::cout << "Resource not added: no such node - " << node << std::endl;
#endif

    return ;
  }
  if (_rt.addResource (node, name, priority)) {
#ifdef DEBUG
    std::cout << "Resource added: " << std::endl << "\t" <<
    node << std::endl << "\t" << name <<
    std::endl << "\t" << priority << std::endl;
#endif

    recalculateState(name);
  }
}

void Manager::receiveChangeResource(const string& message) {
  std::istringstream m(message);
  string node;
  string name;
  int priority;

  m >> node >> name >> priority;
  if (std::find(_groupMembers.begin(), _groupMembers.end(), node) ==
      _groupMembers.end()) {
#ifdef DEBUG
    std::cout << "Resource not changed: no such node - " << node << std::endl;
#endif

    return ;
  }
  if (_rt.changeResource (node, name, priority)) {
#ifdef DEBUG
    std::cout << "Resource changed: " << std::endl << "\t" <<
    node << std::endl << "\t" << name <<
    std::endl << "\t" << priority << std::endl;
#endif

    recalculateState(name);
  }
}

void Manager::receiveDelResource(const string& message) {
  std::istringstream m(message);
  string node;
  string name;

  m >> node >> name;
  if ((node == _privateGroup) && (_rt.getBestNode(name) == _privateGroup))
    _rt.deactivate(name, node);
  if (_rt.delResource (node, name)) {
#ifdef DEBUG
    std::cout << "Resource deleted: " << std::endl << "\t" <<
    node << std::endl << "\t" << name << std::endl;
#endif

    recalculateState(name);
  }
}

void Manager::recalculateState(const string& name) {
  if (name.size()) {
    string node = _rt.getBestNode(name);
    if (node.size())
      if (node == _privateGroup)
        _rt.activate(name, node);
      else
        _rt.deactivate(name, _privateGroup);
  }
  else {
    ResTable table = _rt.getResTable();
    for_each(table.begin(), table.end(), RecalculateState(this));
  }
}

std::ostream& operator<< (std::ostream& s, const Manager& m) {
  return s << m._rt;
}

} // end of namespace
