#include "resourcetable.h"

#include <algorithm>
#include <iterator>

using std::string;
using std::find;
using std::remove_if;
using std::find_if;

namespace ClusterManagement {

u_int16_t ResourceTable::getPriority (const string& node, const string& name) const {
  Rcit i = find (_rTable.begin(), _rTable.end(), Resource(node, name, 0));

  if ( i != _rTable.end())
    return i->getPriority();
  return 0;
}

struct RTSort {
  bool operator () (Resource r1, Resource r2) const {
    if (r1.getPriority() == r2.getPriority())
      if (r1.getName() == r2.getName())
        return r1.getNode() > r2.getNode();
      else
        return r1.getName() > r2.getName();
    return r1.getPriority() > r2.getPriority();
  }
};

void ResourceTable::sort () {
  if (!_isSorted) {
    _rTable.sort(RTSort());
    _isSorted = true;    
  }
}

string ResourceTable::getBestNode (const string& name) {
  _rTable.sort(RTSort());
  for (Rcit r = _rTable.begin(); r != _rTable.end(); r++)
    if (r->getName() == name)
      return r->getNode();
  return "";
}

bool ResourceTable::addResource (const Resource& resource) {
  if ( find(_rTable.begin(), _rTable.end(), resource) == _rTable.end() ) {
    _rTable.push_back(resource);
    _isSorted = false;
    return true;
  }
  return false;
}

bool ResourceTable::addResource (const string& node, const string& name, u_int16_t priority) {
  return addResource (Resource(node, name, priority));
}

bool ResourceTable::changeResource (const string& node, const string& name, u_int16_t priority) {
  Resource resource = Resource(node, name, priority);
  Rit rit = find(_rTable.begin(), _rTable.end(), resource);

  if (rit != _rTable.end() ) {
    rit->setPriority(priority);
    _isSorted = false;
    return true;
  }
  return false;
}

bool ResourceTable::delResource (const string& node, const string& name) {
  size_t size = _rTable.size();
  _rTable.remove (Resource(node, name, 0));
  return _rTable.size() < size;
}

void ResourceTable::delResource (const string& name) {
  _rTable.remove_if (ResourceNameCompare(name));
}

void ResourceTable::delNode (const string& node) {
  _rTable.remove_if (NodeNameCompare(node));
}

void ResourceTable::activate (const string& name, const string& node) {
  Rit r = find_if (_rTable.begin(), _rTable.end(), ResourceCompare(name, node));
  if (r != _rTable.end())
    r->activate();
}

void ResourceTable::deactivate (const string& name, const string& node) {
  Rit r = find_if (_rTable.begin(), _rTable.end(), ResourceCompare(name, node));
  if (r != _rTable.end())
    r->deactivate();
}

std::ostream& operator<< (std::ostream& s, const ResourceTable& rt) {
  ResTable resTable = rt.getResTable();
  copy(resTable.begin(), resTable.end(), std::ostream_iterator<Resource>(s, "\n"));
  return s;
}

}
