#!/usr/bin/perl

# Never use "-w" option in production version - may cause problems with EXMH
# (due to the way tcl interprets messages printed to stderr)

# written by Jason Daniel Rennie <jr6b+@andrew.cmu.edu>

# ifilter.mh is a wrapper for ifile which is specific to the MH
# mail system.  It accepts a message to be filtered on standard input
# and stores that message using rcvstore to the location determined by
# ifile.

####################
# global variables #
####################

# If the MH binary directories are not in your PATH, you will need to
# change "rcvstore" so that it is a fully-qualified path
$mh_rcvstore = "rcvstore";

# If the ifile binary is not in your path, give the fully qualified path below
$ifile_binary = "ifile";

# additional arguments to pass to ifile (esp. lexing options)
$ifile_args = "-h";

# indication of whether we should create tmp file
$tmp2_file = 0;

################
# main program #
################

&parse_args();

if ($tmp2_file)
{
    $file = $ENV{'HOME'}."/.ifilter.log";
    open(TMP2, "> $file");
    chmod 0600, $file;
}

# Checks to see if $mh_rcvstore can be executed from a shell
$executable = "false";
if (! -x $mh_rcvstore) {
    foreach $dir (split(":", $ENV{"PATH"})) {
	if (-x $dir."/".$mh_rcvstore) {
	    $executable = "true";
	}
    }
    if ($executable eq "false") 
    {
	if ($tmp2_file) {
	    print TMP2 "$mh_rcvstore is not executable\n";
	}
	die "$mh_rcvstore is not executable\n";
    }
}

foreach $arg (@ARGV) {
    if ($arg =~ /^--help/ || $arg =~ /^-h/) { &print_usage; }
}

################################
# gets environment information #
################################

if ($tmp2_file) {
    print TMP2 "Reading environment information\n";
}

$home_dir = $ENV{"HOME"}."/";
if ($tmp2_file) {
    print TMP2 "home directory is $home_dir\n";
}

open(MH, "< $home_dir.mh_profile");

while (<MH>) {
    if (m/^path:\s*(\S+)/i) {
	$mail_path = $1;
	$mail_path .= "/";
	# if mail path is not absolute, make it so
	if ($mail_path =~ m/^[^\/]/) {
	    $mail_path = $home_dir.$mail_path;
	}
    }
    if (m/^draft-folder:\s*(\S+)/i) {
	$draft_folder = $1;
    }
    if (m/^context:\s*(\S+)/i) {
	$context_file = $1;
	# if file location is not absolute, make it so
	if ($context_file =~ m/^[^\/]/) {
	    $context_file = $mail_path.$context_file;
	}
    }
    if (m/^inbox:\s*(\S+)/i) {
	$inbox_folder = $1;
    }
}

if (!$context_file) {
    $context_file = $mail_path."context";
}
if (!$inbox_folder) {
    $inbox_folder = "inbox";
}

#############################################
# updates accuracies file ~/.idata_accuracy #
#############################################

if ($tmp2_file) {
    print TMP2 "Updating accuracies file\n";
}

$acc_file = $home_dir.".idata_accuracy";

$filters = 1;
$refiles = 0;

if (open(ACC, $acc_file))
{
    $line = <ACC>;
    $line =~ m/filters\s*=\s*(\d*)\s*refiles\s*=\s*(\d*)/;

    $filters += $1;
    $refiles += $2;

    $accuracy = int(($filters - $refiles)/$filters * 10000)/100;

    close(ACC);
}

if (open(ACC, "> $acc_file"))
{
    print ACC "filters = $filters  refiles = $refiles\n";
    if ($accuracy) { print ACC "Accuracy = $accuracy \%\n"; }
    close(ACC);
} else {
    print TMP "Was not able to write accuracies file\n";
}

##########################################################
# pipes message on stdin to ifile program, parses output #
##########################################################

if ($tmp2_file) {
    print TMP2 "Calling ifile program\n";
}

# Temporary file for storing a copy of incoming message
$tmp_file = "/tmp/ifile".(time() % 100000);

open(TMP, "> $tmp_file")
    || die "Could not open $tmp_file: $!\n";
@message = <STDIN>;
print TMP @message;
close(TMP);
chmod(0600, $tmp_file);

$command = "$ifile_binary $ifile_args --query-insert $tmp_file |";
open(IFILE, $command)
    || die "Could not execute \"$command\": $!\n";
@query_results = <IFILE>;
close(IFILE);

unlink $tmp_file;

# Temporary file for testing the writability of a directory
$tmp_file = ".ifile".(time() % 100000);

## find the best matching folder that doesn't have a .skip_me file
$best = shift(@query_results);
# print "Trying $best folder\n";
$best =~ m/^(\S+)/;
$best_folder = $1;

# Use the first folder which doesn't have a .skip_me file and which we
# can write to.
while ((-f $mail_path.$best_folder."/.skip_me" && (@query_results > 0))
       || !(open(FOO, "> ".$mail_path.$best_folder."/".$tmp_file))) {
    $best = shift(@query_results);
    # print "Trying folder $best";
    if ($best =~ m/^-+$/) { last; }
    $best =~ m/^(\S+)/;
    $best_folder = $1;
}

if (@query_results <= 0) {
    $best_folder = $inbox_folder;
} else {
    close(FOO);
    unlink($mail_path.$best_folder."/".$tmp_file);
}

#######################################################
# use MH rcvstore to put message into selected folder #
#######################################################

if ($tmp2_file) {
    print TMP2 "Calling MH rcvstore to store message\n";
}

open(VER, "$ifile_binary --version |")
    || die "Error: could not execute $ifile_binary: $!\n";
$version = <VER>;
chop $version;
close(VER);

if ($tmp2_file) {
    print TMP2 "$version => $best_folder\n";
}

open(RCV, "| $mh_rcvstore +$best_folder")
    || die "Error: could not execute rcvstore command: $!\n";
$print_filter = 0;
foreach $line (@message)
{
    # Add the X-filter header at the end of the header section
    if ($line =~ m/^\n$/ && (!$print_filter))
    {
	print RCV "X-filter: $version => $best_folder\n";
	$print_filter = 1;
    }
    # Rename old x-filter header if we are still in header section
    $line =~ s/^x-filter: /Old-x-filter: /i if (!$print_filter);
    # Make sure each line ends with feed-line
    $line .= "\n" if ($line !~ m/\n$/);

    print RCV $line;
}
# Print X-filter header now if we never did so before
print RCV "X-filter: $version => $best_folder\n" if (!$print_filter);
close(RCV);

if ($tmp2_file) {
    close(TMP2);
}

###############
# subroutines #
###############

sub print_usage
{
    print "Usage: ifilter.mh < <FILE>\n";
    print "    or cat <FILE> | ifilter.mh\n";
    print "\n";
    print "\t-g\tCreate ~/.ifilter.log file for debugging purposes\n";
    print "\t-h\tDisplay usage information\n";
    print "\n";
    print "ifilter.mh accepts a text file on standard input, uses ifile\n";
    print "to compute the most appropriate destination folder and then uses\n";
    print "MH rcvstore to store the message in the chosen folder.\n";
    exit 0;
}


sub parse_args
{
    while (@ARGV > 0)
    {
	$arg = shift(@ARGV);
	
	if ($arg eq "-g" || $arg eq "--log-file")
	{
	    $tmp2_file = 1;
	}
	elsif ($arg eq "-h" || $arg eq "--help")
	{
	    &print_usage();
	}
    }
}
