#!/usr/bin/perl

# Never use "-w" option in production version - may cause problems with EXMH
# (due to the way tcl interprets messages printed to stderr)

# written by Jason Daniel Rennie <jr6b+@andrew.cmu.edu>

# irefile.mh is a wrapper for MH refile which passes information to the
# ifile executable to update the e-mail database

# Note: when dealing with EXMH, messages should be printed to STDERR only
#       if you want to have EXMH consider the execution a failure.

####################
# global variables #
####################

# If the MH binary directories are not in your PATH, you will need to
# change "refile" so that it is a fully-qualified path
$mh_refile = "refile";

# If the ifile binary is not in your path, give the fully qualified path below
$ifile_binary = "ifile";

# additional arguments to pass to ifile (esp. lexing options)
$ifile_args = "-h";

# indication of whether we should create tmp file
$tmp_file = 0;

################
# main program #
################

&parse_args(@ARGV);

if ($tmp_file)
{
    $file = $ENV{'HOME'}."/.irefile.log";
    open(TMP2, "> $file");
    chmod 0600, $file;
    print TMP2 join(' ', @ARGV)."\n";
}

# Checks to see if $mh_rcvstore can be executed from a shell
$executable = "false";
if (! -x $mh_refile) {
    foreach $dir (split(":", $ENV{'PATH'})) {
	if (-x $dir."/".$mh_refile) {
	    $executable = "true";
	}
    }
    if ($executable eq "false")
    {
	if ($tmp_file) {
	    print TMP2 "$mh_refile is not executable\n";
	}
	die "$mh_refile is not executable\n";
    }
}

foreach $arg (@ARGV) {
    if ($arg =~ /^--help/ || $arg =~ /^-h/) { &print_usage; }
}

################################
# gets environment information #
################################

if ($tmp_file) {
    print TMP2 "Reading environment information\n";
}

$home_dir = $ENV{"HOME"}."/";

open(MH, "$home_dir.mh_profile")
    || &quick_finish("Not able to open \"$home_dir.mh_profile\": $!\n");
while (<MH>) {
    if (m/^path:\s*(\S+)/i) {
	$mail_path = $1;
	$mail_path .= "/";
	# if mail path is not absolute, make it so
	if ($mail_path =~ m/^[^\/]/) {
	    $mail_path = $home_dir.$mail_path;
	}
    }
    if (m/^draft-folder:\s*(\S+)/i) {
	$draft_folder = $1;
    }
    if (m/^context:\s*(\S+)/i) {
	$context_file = $1;
	# if file location is not absolute, make it so
	if ($context_file =~ m/^[^\/]/) {
	    $context_file = $mail_path.$context_file;
	}
    }
    if (m/^inbox:\s*(\S+)/i) {
	$inbox_folder = $1;
    }
}

if (!$context) {
    $context_file = $mail_path."context";
}
if (!$inbox_folder) {
    $inbox_folder = "inbox";
}
close(MH);

open(CONTEXT, $context_file)
    || &quick_finish("Not able to open \"$context_file\": $!\n");
while (<CONTEXT>) {
    if (m/current-folder:\s*(\S+)/i) {
	$current_folder = $1;
    }
}
close(CONTEXT);

#################################
# parses command line arguments # 
#################################

if ($tmp_file) {
    print TMP2 "Parsing command line arguments\n";
}

&parse_args();

if (! $source_folder) {
    $source_folder = $current_folder;
}

if (! $dest_folder) {
    print STDERR "I need a destination folder!\n";
    &print_usage;
}

if ($tmp_file) {
    print TMP2 "Messages to refile: ".join(' ', @orig_message)."\n";
}

#######################
# fix up message list #
#######################

# if the '-file' command line option was used, disregard any message names
#   included on the command line.  Warn the user if '-src' or message names
#   were included as command line options

if ($source_file) {
    if (($msg_count = @orig_message) > 0) {
	print "Messages other than $source_file are being ignored.\n";
    }
    if ($source_folder) {
	print "\'-src\' command line option is being ignored.\n";
	$source_folder = "";
    }
    $#orig_message = -1;
    $orig_message[0] = $source_file;
}

#############################################
# updates accuracies file ~/.idata_accuracy #
#############################################

if ($tmp_file) {
    print TMP2 "Updating accuracies file\n";
}

$acc_file = $home_dir.".idata_accuracy";

$filters = 0;
$refiles = @orig_message;

if (open(ACC, $acc_file))
{
    $line = <ACC>;
    $line =~ m/filters\s*=\s*(\d*)\s*refiles\s*=\s*(\d*)/;
    
    $filters += $1;
    $refiles += $2;
    if ($filters > 0) {
	$accuracy = int(($filters - $refiles)/$filters * 10000)/100;
    } else {
	$accuracy = 0.0;
    }
    if ($accuracy < 0.0) {
	$accuracy = 0.0;
    }
    close(ACC);
}

if (open(ACC, "> $acc_file"))
{
    print ACC "filters = $filters  refiles = $refiles\n";
    if ($accuracy) { print ACC "Accuracy = $accuracy \%\n"; }
    close(ACC);
} else {
    print STDERR "Was not able to write accuracies file\n";
}

####################################
# passes messages to ifile program #
####################################

if ($tmp_file) {
    print TMP2 "Calling ifile program\n";
}

# creates messages with absolute path names
foreach $message (@orig_message) {
    $absolute_msg[@absolute_msg] = $mail_path.$source_folder."/".$message;
}

# create the ifile command line
if ($tmp_file)
{
    $command = "$ifile_binary $ifile_args -g ";
}
else
{    
    $command = "$ifile_binary $ifile_args ";
}
if (! -f $mail_path.$source_folder."/.skip_me") {
    $command .= "--delete=$source_folder ";
}
if (! -f $mail_path.$dest_folder."/.skip_me") {
    $command .= "--insert=$dest_folder ";
}
$command .= join (" ", @absolute_msg);

if ($tmp_file) {
    print TMP2 $command."\n";
}

open(FP, "$command |")
    || &quick_finish("Could not execute \"$arg\": $!\n");
while (<FP>) {print;}
close(FP);

#########################################################
# uses MH refile to move messages to destination folder #
#########################################################

if ($tmp_file) {
    print TMP2 "Calling MH refile to move messages\n";
}

$command = $mh_refile;
while (@ARGV > 0)
{
    $arg = shift(@ARGV);
    if ($arg eq "-g" || $arg eq "--tmp-file")
    {
	next;
    }
    $command .= " ".$arg;
}

if ($tmp_file) {
    print TMP2 $command."\n";
}
open(FP, "$command |")
    || &undo_ifile("Could not execute \"$command\": $!\n");
while (<FP>) {print;}
close(FP);

if ($tmp_file) {
    close(TMP2);
}

#######################
# --- subroutines --- #
#######################

# just do the refile, forget about updating the database
sub quick_finish
{
    print @_;
    $command = "$mh_refile ".join(" ", @ARGV);
    system $command
	|| die "Could not execute \"$arg\": $!\nI give up!\n";
}

# At this point, stats have been added to the database for the messages passed
# in on the command line.  This function reverses that effect.
sub undo_ifile
{
    $error_message = $_[0];
    
    # create the ifile command line
    $command = "$ifile_binary $ifile_args ";
    if (! -f $mail_path.$source_folder."/.skip_me") {
	$command .= "--insert=$source_folder ";
    }
    if (! -f $mail_path.$dest_folder."/.skip_me") {
	$command .= "--delete=$dest_folder ";
    }
    $command .= join (" ", @absolute_msg);
    
    system $command
	|| die "ifile database modified without refile execution\n$error_message";

    die $error_message;
}

sub print_usage
{
    print "Usage: irefile.mh <MH_OPTIONS>\n";
    print "\n";
    print "\t-g\tCreate ~/.irefile.log file for debugging purposes\n";
    print "\t-h\tDisplay usage information\n";
    print "\n";
    print "Uses ifile to update the existing database and then uses MH\n";
    print "refile to move the given message to the indicated mailbox\n";
    exit 0;
}


sub parse_args
{
    local(@argv) = @_;
    local($arg);
    
    while (@argv > 0)
    {
	$arg = shift(@argv);
	# print "arg = $arg\n";

	if ($arg eq "-g" || $arg eq "--tmp-file")
	{
	    $tmp_file = 1;
	}
	elsif ($arg eq "-h" || $arg eq "--help")
	{
	    &print_usage;
	}
	elsif ($arg eq "-src")
	{
	    if (@argv < 1)
	    {
		&print_usage;
	    }
	    $source_folder = shift(@argv);
	    if (! ($source_folder =~ s/^\+//))
	    {
		print STDERR "Error: folder name lacks preceeding '+' or is";
		print STDERR " in wrong format.\n";
		&print_usage;
	    }
	    # if folder name is an absolute path, try to strip the mail path
	    if ($source_folder =~ m/^$mail_path(\S+)/) {
	        $source_folder = $1;
	    }
	    # if folder name ends in '/', remove it
	    if ($source_folder =~ m/^(\S+)\/$/) {
		$source_folder = $1;
	    }
	    #print "source folder = $source_folder\n";
	}
	elsif ($arg eq "-file")
	{
	    if (@argv < 1)
	    {
		&print_usage;
	    }
	    $source_file = shift(@argv);
	    #print "source file (-file) = $source_file\n";   
	}
	elsif ($arg eq "-draft" ||
	       $arg eq "-nolink" ||
	       $arg eq "-preserve" ||
	       $arg eq "-nopreserve" ||
	       $arg eq "-normmproc" ||
	       $arg eq "-link")
	{ }
	elsif ($arg eq "-rmmproc")
	{
	    if (@argv < 1)
	    {
		&print_usage;
	    }
	    shift(@argv);
	}
	elsif ($arg =~ m/^\+(\S+)/)
	{
	    $dest_folder = $1;
	    # if folder name is an absolute path, try to strip the mail path
	    if ($dest_folder =~ m/^$mail_path(\S+)/) {
		$dest_folder = $1;
	    }
	    # if folder name ends in '/', remove it
	    if ($dest_folder =~ m/^(\S+)\/$/) {
		$dest_folder = $1;
	    }
	    #print "dest folder = $dest_folder\n";
	}
	elsif ($arg eq "#")
	{
	    # takes care of "refile # +src +dst" calling convention
	    # this isn't in the documentation, but xmh seems to use it
	    if (@argv < $2) {
		&print_usage;
	    }
	    $source_folder = shift(@argv);
	    $dest_folder = shift(@argv);
	    $source_folder =~ s/^\+//;
	    $dest_folder =~ s/^\+//;
	}
	else
	{
	    $orig_message[$#orig_message+1] = $arg;
	}
    }
}
