/*-
 * Copyright (c) 2003 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
/*
 * Simple BPF-based sampling of non-local IP traffic on an ethernet
 * segment.  Unless you are me, the chances are good you'll want
 * to tweak the definition of "local" in local() to match your own
 * local subnet traffic.  Samples are dropped in the "data/"
 * subdirectory and must be later collated.
 */

#include <sys/types.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <net/bpf.h>
#include <net/ethernet.h>
#include <net/if.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <arpa/inet.h>

#include <fcntl.h>
#include <pcap.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

struct timeval next_bill;

struct billing_entry {
	struct billing_entry	*be_next;
	struct in_addr		 be_addr;
	long long		 be_cost;
};

#define BE_HASHSIZE	256
#define	BE_HASH(a)	((((a).s_addr & 0x000000ff) ^			\
			 ((a).s_addr & 0x0000ff00 >> 8) ^		\
			 ((a).s_addr & 0x00ff0000 >> 16) ^		\
			 ((a).s_addr & 0xff000000 >> 24)) % BE_HASHSIZE)

struct billing_entry *be_hash[BE_HASHSIZE];

static struct bpf_insn	instructions[] = {
/* (000) ldh      [12]                            */
	{ 0x28, 0, 0, 0x0000000c },
/* (001) jeq      #0x800           jt 2    jf 3   */
	{ 0x15, 0, 1, 0x00000800 },
/* (002) ret      #1500                           */
	{ 0x6, 0, 0, 0x000005dc },
/* (003) ret      #0                              */
	{ 0x6, 0, 0, 0x00000000 },
};
static int	num_instructions = sizeof(instructions) /
		    sizeof(struct bpf_insn);

#define	BUFLEN	1500

int
bpf_open(int bufsize, char *ifname)
{
	struct bpf_program	bpf;
	struct timeval		tv;
	struct ifreq		ifr;
	char			device[sizeof("/dev/bpf000")];
#ifdef IMMEDIATE
	u_int			data;
#endif
	int			fd, n = 0;

	do {
		sprintf(device, "/dev/bpf%d", n++);
		fd = open(device, O_RDWR);
	} while (fd < 0 && errno == EBUSY && n < 1000);

	if (fd < 0) {
		perror("open");
		return (-1);
	}

	if (ioctl(fd, BIOCSBLEN, &bufsize)) {
		perror("BIOCSBLEN");
		close(fd);
		return (-1);
	}

	strncpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
	if (ioctl(fd, BIOCSETIF, &ifr)) {
		perror("BIOCSETIF");
		close(fd);
		return (-1);
	}

#ifdef IMMEDIATE
	data = 1;
	if (ioctl(fd, BIOCIMMEDIATE, &data) < 0) {
		perror("BIOCIMMEDIATE");
		close(fd);
		return (-1);
	}
#endif

	tv.tv_sec = 1;
	tv.tv_usec = 0;
	if (ioctl(fd, BIOCSRTIMEOUT, &tv) < 0) {
		perror("BIOCSRTIMEOUT");
		close(fd);
		return (-1);
	}

	if (ioctl(fd, BIOCPROMISC, NULL)) {
		perror("BIOCPROMISC");
		close(fd);
		return (-1);
	}

	bpf.bf_len = num_instructions;
	bpf.bf_insns = instructions;
	if (ioctl(fd, BIOCSETF, &bpf)) {
		perror("BIOCSETF");
		close(fd);
		return (-1);
	}

	return (fd);
}

int
local(struct in_addr addr)
{
	struct in_addr network_addr;

	if (inet_aton("204.156.12.0", &network_addr)) {
		if ((ntohl(network_addr.s_addr) & 0xffffff00) ==
		    (ntohl(addr.s_addr) & 0xffffff00))
			return (1);
	}
	if (inet_aton("206.239.198.0", &network_addr)) {
		if ((ntohl(network_addr.s_addr) & 0xffffff00) ==
		    (ntohl(addr.s_addr) & 0xffffff00))
			return (1);
	}

	return (0);
}

void
bill_init(void)
{
	int i;

	for (i = 0; i < BE_HASHSIZE; i++)
		be_hash[i] = NULL;

	gettimeofday(&next_bill, NULL);
	next_bill.tv_sec += SAMPLING_INTERVAL;
}

struct billing_entry *
be_find(struct in_addr addr)
{
	struct billing_entry *be;

	for (be = be_hash[BE_HASH(addr)]; be != NULL; be = be->be_next)
		if (be->be_addr.s_addr == addr.s_addr)
			break;

	if (be == NULL) {
		be = (struct billing_entry *) malloc(sizeof(*be));
		if (be == NULL) {
			perror("malloc");
			exit(-1);
		}
		be->be_addr = addr;
		be->be_next = be_hash[BE_HASH(addr)];
		be_hash[BE_HASH(addr)] = be;
		be->be_cost = 0;
	}

	return (be);
}

void
bill(struct in_addr addr, size_t len)
{
	struct billing_entry *be;

	be = be_find(addr);
	be->be_cost += len;

#ifdef DEBUG
	printf("Billing %s for %d\n", inet_ntoa(addr), len);
#endif
}

void
bill_dump(void)
{
	struct billing_entry *be;
	struct timeval timestamp;
	char timebuf[128], fnamebuf[128];
#ifdef VERBOSE_NAMES
	struct tm *tm;
#endif
	FILE *output;
	int i;

	gettimeofday(&timestamp, NULL);
#if VERBOSE_NAMES
	tm = localtime((time_t *)&timestamp.tv_sec);
	snprintf(timebuf, 128, "%02d-%02d-%02d-%02d-%02d-%02d",
	    tm->tm_year + 1900, tm->tm_mon, tm->tm_mday, tm->tm_hour,
	    tm->tm_min, tm->tm_sec);
#else
	snprintf(timebuf, 128, "%lu", timestamp.tv_sec);
#endif
	snprintf(fnamebuf, 128, "data/%s", timebuf);

	output = fopen(fnamebuf, "a");
	if (output == NULL) {
		perror(fnamebuf);
		return;
	}

	fprintf(output, "%s\n", timebuf);

	for (i = 0; i < BE_HASHSIZE; i++) {
		for (be = be_hash[i]; be != NULL; be = be->be_next) {
			if (be->be_cost != 0) {
				fprintf(output, "%s:%llu\n",
				    inet_ntoa(be->be_addr), be->be_cost);
#ifdef DEBUG
				printf("%s:%llu\n", inet_ntoa(be->be_addr),
				    be->be_cost);
#endif
				be->be_cost = 0;
			}
		}
	}

	fclose(output);
}

void
bill_maybe_dump(void)
{
	struct timeval timestamp;

	gettimeofday(&timestamp, NULL);
	if (timestamp.tv_sec >= next_bill.tv_sec) {
		next_bill.tv_sec += SAMPLING_INTERVAL;
		bill_dump();
	}
}

void
gather(struct in_addr src, struct in_addr dst, size_t len)
{
	int local_src, local_dst;

	local_src = local(src);
	local_dst = local(dst);

	/* Ignore entirely local traffic. */
	if (local_src && local_dst)
		return;

	/* Ignore bogus entirely non-local traffic. */
	if (!local_src && !local_dst)
		return;

	if (local_src)
		bill(src, len);
	else
		bill(dst, len);
}

void
process_bpf(int bpf_fd)
{
	struct bpf_hdr *hdr;
	char buffer[BUFLEN];
	struct ip *ip;
	size_t len;
	char *bp, *ep;

	while (1) {
		do {
			len = read(bpf_fd, buffer, BUFLEN);
		} while (len == -1 && errno == EINTR);

		if (len < 0) {
			perror("read");
			continue;
		}

		bp = buffer;
		ep = bp + len;

		while (bp < ep) {
			hdr = (struct bpf_hdr *) bp;
			len = hdr->bh_caplen;

			if (len < sizeof(struct ip) +
			    sizeof(struct ether_header)) {
				perror("truncated packet");
				continue;
			}

			ip = (struct ip *) ((void *)hdr + hdr->bh_hdrlen +
			    sizeof(struct ether_header));
#if 0
			printf("%s to ", inet_ntoa(ip->ip_src));
			printf("%s\n", inet_ntoa(ip->ip_dst));
#endif

			gather(ip->ip_src, ip->ip_dst, len);

			bp += BPF_WORDALIGN(hdr->bh_caplen + hdr->bh_hdrlen);
		}

		bill_maybe_dump();
	}
}

int
main(int argc, char *argv[])
{
	int bpf_fd;

	bpf_fd = bpf_open(BUFLEN, IFNET);
	if (bpf_fd == -1)
		exit(-1);

	bill_init();
	process_bpf(bpf_fd);

	exit(0);
}
